<?php

namespace App\Models\Concerns;

use App\DataTable\DataTableConfiguration;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Query\JoinClause;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;

/**
 * Trait HasSort.
 *
 *
 * @method Builder sort( Request $request = null ) @see HasSort::scopeSort()
 */
trait HasSort
{
    // public function print($string)
    // {
    //     fwrite(STDERR, "\n>>\n".$string."\n");
    // }
    //
    // public function inspect($ob)
    // {
    //     $this->print(Inspect::get($ob));
    // }
    //
    // public function header_print($string, $val)
    // {
    //   @$this->print("X-".$string.": ". $val);
    //   @header("X-".$string.": ". $val);
    // }

    /**
     * @throws Exception
     */
    public function scopeSort(Builder $builder, mixed $request = null)
    {

        if (is_null($request)) {
            $request = request();
        }

        // get sort columns as array
        if ($request instanceof Request) {
            $sortObjs = json_decode($request->input('sortObjs', '{}'), true);
        } else {
            $sortObjs = $request['sortObjs'] ?? [];
        }

        // $this->header_print("sortObjs", json_encode($sortObjs));
        // $this->header_print("sortableColumns", json_encode($this->sortableColumns("")));
        // $this->header_print("filterableColumns", json_encode($this->filterableColumns("")));

        // apply sort on columns
        if (! empty($sortObjs)) {
            foreach ($sortObjs as $sortObj) {
                if (0 || in_array($sortObj['column'], $this->sortableColumns($sortObj['column']))) {
                    $key = DataTableConfiguration::getRealKey($this->getDataTableClass($builder), $sortObj['column']);

                    // $this->header_print("sort-".$sortObj['column']."-realKey", json_encode($key));
                    $key = is_array($key) ? $key['key'] : $key;

                    $relation = $this->isRelationSort($key);
                    if ($this->sortScopeDefined($relation)) {
                        $method = 'scope'.$this->getScopeRelationNameSort($relation, '', 'sort');
                        $this->$method($builder, $relation, $sortObj['ascending']);
                    } else {
                        $this->sortByKey($builder, DataTableConfiguration::getRealKey($this->getDataTableClass($builder), $sortObj['column']), $sortObj['ascending'] ?? true);
                    }
                }
            }
        } else {
            // No sort override, we use the default sort.
            $builder->defaultSort();
        }
    }

    public function scopeDefaultSort(Builder $builder, $request = null)
    {
        // default
        $tableSpecifications = DataTableConfiguration::getTableSpecifications($this->getDataTableKey($builder), false)['table_specifications'];

        if (! $tableSpecifications) {
            return;
        } // If there's no specifications, we abort the sort

        $defaultSorts = $tableSpecifications['default_sort'];

        foreach ($defaultSorts as $field) {
            $this->sortByKey($builder, DataTableConfiguration::getRealKey($this->getDataTableKey($builder), $field['field']), $field['sort'] == 'ASC');
        }
    }

    protected function sortScopeDefined($relation)
    {
        if (! $relation) {
            return false;
        }

        // $this->header_print("sortScopeDefined-relation", json_encode($relation));

        $method = 'scope'.$this->getScopeRelationNameSort($relation, '', 'sort');
        // $this->header_print("scope-method", $method);

        if (method_exists($this, $method)) {
            return true;
        }

        return false;
    }

    /**
     * Scope to sort an individual key.
     *
     *
     * @return Builder|\Illuminate\Database\Query\Builder
     */
    public function scopeSortKey(Builder $builder, $key, bool $ascending = true)
    {
        return $this->sortByKey($builder, $key, $ascending);
    }

    /**
     * Main function to sort a key.
     *
     *
     * @return Builder|\Illuminate\Database\Query\Builder
     *
     * @throws Exception
     */
    public function sortByKey(Builder $builder, $key, bool $ascending = true)
    {
        // check is a relational key?
        if (is_array($key)) {
            $relation = $key['is_relation'] ? $this->isRelationSort($key['key']) : false;
            if ($relation) {
                $relation['base'] = $key['base'] ?? true;
            }
        } else {
            $relation = $this->isRelationSort($key);
        }
        $realKey = $key['key'] ?? $key;

        if ($relation) {
            $scopeName = 'Sort'.ucfirst($relation['name']);
            if (method_exists($this, 'scope'.$scopeName)) {
                if (($result = $builder->{$scopeName}($relation, $ascending)) !== false) {
                    return $result;
                }
            }

            $relationInstance = $builder->getModel()->{$relation['name']}();
            if ($relationInstance instanceof BelongsTo) {
                return $this->sortBelongsToRelations($builder, $relation, $ascending);
            } else {
                throw new Exception('you should define a new scope to implement this relation: '.$relation['name']);
            }
        }

        if (($key['from_json'] ?? false)) {
            return $builder->orderBy($realKey, $ascending ? 'asc' : 'desc');
        }

        // $this->header_print("XXXXXXXXXXXXXXXXXXXXXXX","1");
        return $builder->orderByRaw($this->getSortingSQLQuery($this, $realKey, $ascending));
    }

    /**
     * Get sql query to sort a key.
     */
    public function getSortingSQLQuery(Model $model, string $key, bool $ascending = true): string
    {
        // check calculated columns to sort without table name
        if (method_exists($this, 'calculatedColumns') && in_array($key, $this->calculatedColumns())) {
            $realKey = '`'.$key.'`';
        } else {
            $realKey = '`'.str_replace('.', '`.`', $model->qualifyColumn($key)).'`';
        }

        if ($ascending) {
            $attributeName = last(explode('.', $key));
            if (($model->getCasts()[$attributeName] ?? 'string') == 'string') {
                $sqlQuery = 'COALESCE('.$realKey.", 'zz') ASC";
            } else {
                $sqlQuery = $realKey.' ASC';
            }
        } else {
            $sqlQuery = $realKey.' DESC';
        }

        // $this->header_print("Sorting-SQL", $sqlQuery);

        return $sqlQuery;
    }

    public function isTableJoined(Builder $builder, $table): bool
    {
        $joins = array_map(function (JoinClause $join) {
            return $join->table;
        }, $builder->getQuery()->joins ? $builder->getQuery()->joins : []);

        return in_array($table, $joins);
    }

    private function getDataTableKey(Builder $builder)
    {
        if ($builder->getModel()->dataTableKey) {
            return $builder->getModel()->dataTableKey;
        }

        return get_class($builder->getModel());
    }

    /**
     * Is key from relation (check by "." between key).
     *
     *
     * @return array|bool <false> if not relation
     */
    public function isRelationSort($key)
    {
        $count = count($relation = explode('.', $key));
        if ($count > 1) {
            // the relation name and the key inside the relation
            return [
                'name' => $relation[0],
                'key' => $relation[1],
                'subkey' => isset($relation[2]) ? $relation[2] : null,
                'direct_relation' => $count == 2,
                'combined_key' => $key,
            ];
        }

        return false;
    }

    private function getScopeRelationNameSort(array $relation, string $operator, $for = 'filter')
    {
        $prefix = ($for == 'filter') ? 'Filter' : 'Sort';
        static $count = 0;
        $count++;
        // @header('X-scope-relation-'.$count.': '.$prefix.ucfirst($relation['name']));

        return $prefix.ucfirst($relation['name']);
    }

    private function isRelationsByJoinSort()
    {
        return property_exists($this, 'relationsByJoin') ? $this->relationsByJoin : false;
    }

    /**
     * Sort on belongs to relations
     *
     * @throws Exception
     */
    protected function sortBelongsToRelations(Builder $builder, array $relationInfo, $ascending): Builder
    {
        $relations = explode('.', $relationInfo['combined_key']);
        $key = Arr::last($relations);

        $parent = $builder->getModel();
        foreach (array_slice($relations, 0, -1) as $relationName) {
            $relationInstance = $parent->{$relationName}();
            if (! $relationInstance instanceof BelongsTo) {
                throw new Exception('you should define a new scope to implement this relation: '.$relationInfo['name']);
            }

            if (! $this->isTableJoined($builder, $relationInstance->getRelated()->getTable())) {
                $builder->leftJoin($relationInstance->getRelated()->getTable(), $relationInstance->getQualifiedForeignKeyName(), '=', $relationInstance->getQualifiedOwnerKeyName());
            }

            $parent = $relationInstance->getRelated();
        }

        $baseTable = $builder->getModel()->getTable();

        return $builder->select("{$baseTable}.*")->orderBy($parent->qualifyColumn($key), $ascending ? 'asc' : 'desc');
    }
}
