<?php

namespace App\Models\Concerns;

use App\DataTable\DataTableConfiguration;
use App\Helpers;
use App\Lib\SphinxSearch\SphinxSearch;
use App\Models\Setting;
use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\Relations\MorphToMany;
use Illuminate\Database\Query\Expression;
use Illuminate\Database\Query\JoinClause;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

/**
 * Trait HasFilters.
 *
 *
 * @method Builder|$this filter( Request|array $request = null ) @see HasFilters::scopeFilter()
 * @method Builder|$this addRelations( Request|array $request = null ) @see HasFilters::scopeAddRelations()
 * @method Builder|$this addCustomColumns() @see HasFilters::scopeAddCustomColumns()
 */
trait HasFilters
{
    /**
     * Get operators or the function's operator.
     *
     * @param  null  $operator  <NULL> to return all operators
     * @return array|string
     * @throws Exception
     */
    public function operators($operator = null): array|string
    {
        $operators = [
            '>=' => 'filterWhereGreaterOrEqual',
            '<=' => 'filterWhereLessOrEqual',
            '>' => 'filterWhereGreater',
            '<' => 'filterWhereLess',
            'isNoneOf' => 'filterWhereAllNotIn',
            'isAnyOf' => 'filterWhereIn',
            '|' => 'filterWhereIn',
            '$' => 'filterWhereIn', // replaced by "&" to retrieve GET request parameters correctly.
            '!=' => 'filterWhereNot',
            '=' => 'filterWhere',
            'isNotEmpty' => 'filterWhereNot',
            'isEmpty' => 'filterWhere',
            'isAssigned' => 'filterWhere',
            'isNotAssigned' => 'filterWhereNot',
            'contains' => 'filterWhereContains',
            'doesNotContain' => 'filterWhereNotContains',
            'isWithin' => 'filterWhereDateBetween',
            'startsWith' => 'filterWhereStartsWith',
            'endsWith' => 'filterWhereEndsWith',
        ];

        if ($operator and array_key_exists($operator, $operators)) {
            return $operators[$operator];
        }

        if (isset($operator)) {
            throw new Exception('Not supported operator.');
        }

        return $operators;
    }

    /**
     * Filter the request.
     *
     * @param  Request|array  $request
     *
     * @throws Exception
     */
    public function scopeFilter(Builder $builder, $request = null): Builder
    {
        $filtersAndQuery = $this->getFiltersAndQuery($request);
        $generalQuery = $filtersAndQuery['query'];
        $builder->where(function (Builder $builder) use ($request) {

            $filtersAndQuery = $this->getFiltersAndQuery($request);
            $filters = $filtersAndQuery['filters'];
            $generalQuery = $filtersAndQuery['query'];
            $conjunction = $filters['conjunction'] ?? 'and';
            $availableColumns = collect($this->availableColumns());
            // apply filter on columns
            foreach ($filters['filterSet'] as $filter) {
                $columnConfig = $availableColumns->where('data_name', $filter['column'])->first();

                if (is_null($columnConfig) && method_exists($this, 'scopeColumns')) {
                    $isScope = in_array($filter['column'], $this->scopeColumns() ?? []);
                } else {
                    $isScope = $columnConfig['is_scope'] ?? false;
                }

                if ($isScope)
                {
                    $scopeFunction = 'Filter' . Str::studly($filter['column']);
                    if (method_exists($this, 'scope' . $scopeFunction)) {
                        $builder->$scopeFunction($filter['operator'] ?? '=', @$filter['value'], $filter['conjunction'] ?? 'and');
                    }
                    continue;
                }
                // @header("X-filter-column-".$filter['column'].": @@@@@");
                if ($this->isFilterableColumn(@$filter['column'])) {
                    // @header("X-filter-column-".$filter['column']."-filterable: TRUE");
                    $operator = $filter['operator'] ?? '=';
                    $key = DataTableConfiguration::getRealKey($this->getDataTableClass($builder), $filter['column']);
                    $value = $this->valueByOperator($operator, isset($filter['value']) && is_string($filter['value']) ? rawurldecode($filter['value']) : $filter['value'] ?? null);

                    // @header("X-filter-column-".$filter['column']."-RealKey: ".$key);

                    if (is_string($key) && $this->hasCast($key) && ! is_array($value) && ! is_null($value)) {
                        $timezone = Helpers::setting(Setting::KEY_DEFAULT_TIMEZONE, config('app.timezone'));

                        switch ($this->getCastType($key)) {
                            case 'datetime':
                                $value = str_replace(' ', '+', $value);
                                $value = Carbon::parse($value, $timezone);
                                break;
                            case 'date':
                                $value = Carbon::parse($value, $timezone)->startOfDay();
                                break;
                            default:
                                $value = $this->castAttribute($key, $value);
                        }
                    }
                    $this->filterKey($builder, $key, $operator, $value, $conjunction);
                }
            }

            $joins = [];
            $groups = [];
            $bindings = null;

            // contains "All Fields"
            if ($generalQuery) {
                $generalQuery = e($generalQuery);

                if ($this->generalSearchByFulltext) {
                    if (SphinxSearch::indexEnabled(env('SPHINX_INDEX_PREFIX', 'sku_').Arr::get($this->fulltextSearchParams, 'index', false))) {
                        if (in_array('sku', $this->generalFilterableColumns())) {
                            $count = $builder->clone()->where('sku', $generalQuery)->count();
                            if ($count > 0) {
                                return $builder->where('sku', $generalQuery);
                            }
                        }

                        $args = empty($this->fulltextSearchParams) ? [] : $this->fulltextSearchParams;
                        $args['query'] = $generalQuery;
                        $builder = SphinxSearch::filter($builder, env('SPHINX_INDEX_PREFIX', 'sku_').Arr::get($this->fulltextSearchParams, 'index'), $args);
                        if (! empty($this->fulltextSearchParams['order_by_raw']) && empty($builder->getQuery()->orders)) {
                            $builder->orderByRaw($this->fulltextSearchParams['order_by_raw'], [$generalQuery]);
                        }
                    } else {
                        $match_score = '  
                        (
                         MATCH('.implode(',', $this->generalFilterableColumns()).') AGAINST(?)
                        + IF(CONCAT_WS(" ", '.implode(',', $this->generalFilterableColumns()).') LIKE ?, 100, 0)
                        
                    ';

                        //Check if sku exists in your filterable columns before adding it to match_score
                        if (in_array('sku', $this->generalFilterableColumns())) {
                            $match_score .= ' + IF(sku = ?, 1000, 0)';
                            $parameters = [$generalQuery, '%'.$generalQuery.'%', $generalQuery];
                        } else {
                            $parameters = [$generalQuery, '%'.$generalQuery.'%'];
                        }

                        $match_score .= ')';

                        $builder->addSelect('*');
                        $builder->selectRaw($match_score.' as MATCH_SCORE , sku', $parameters);
                        $builder->whereRaw($match_score.' > 1 ', $parameters);
                        $builder->orderBy('MATCH_SCORE', 'DESC');
                    }
                } else {
                    $builder->where(function (Builder $builder) use ($generalQuery, &$joins, &$groups, &$bindings) {

                        foreach ($this->generalFilterableColumns() as $key) {
                            $this->filterKey($builder, DataTableConfiguration::getRealKey(get_class($builder->getModel()), $key), 'contains', $generalQuery, 'or');
                        }

                        // $joins  = $builder->getQuery()->joins;
                        // $groups = $builder->getQuery()->groups ?: [];

                        return $builder;
                    });
                }
            }

            $joins = collect($builder->getQuery()->joins)->keyBy('table')->all() + collect($joins)->keyBy('table')->all();

            $builder->getQuery()->joins = array_values($joins);
            $builder->getQuery()->groups = array_unique($groups + ($builder->getQuery()->groups ?: [])) ?: null;

            if ((empty($builder->getQuery()->columns) || count($builder->getQuery()->columns) <= 1)) {
                if ($builder->getQuery()->columns && count($builder->getQuery()->columns) === 1) {
                    if (! is_object($builder->getQuery()->columns[0])) {
                        $builder->select($this->getTable().'.*');
                    }
                } else {
                    $builder->select($this->getTable().'.*');
                }
            }

            return $builder;
        });
        if (
            ! SphinxSearch::indexEnabled(env('SPHINX_INDEX_PREFIX', 'sku_').Arr::get($this->fulltextSearchParams, 'index', false))
            && $generalQuery
            && $this->generalSearchByFulltext) {
            $generalQuery = e($generalQuery);
            if (in_array('sku', $this->generalFilterableColumns())) {
                $builder->orderByRaw('IF(sku = ?, 1000, IF(CONCAT_WS(" ", '.implode(',', $this->generalFilterableColumns()).') LIKE ?, 100, MATCH('.implode(',', $this->generalFilterableColumns()).') AGAINST(?))) DESC', [$generalQuery, '%'.$generalQuery.'%', '%'.$generalQuery.'%' ?? '']);
            } else {
                $builder->orderByRaw('IF(CONCAT_WS(" ", '.implode(',', $this->generalFilterableColumns()).') LIKE ?, 100, MATCH('.implode(',', $this->generalFilterableColumns()).') AGAINST(?)) DESC', ['%'.$generalQuery.'%', '%'.$generalQuery.'%' ?? '']);
            }
        }

        return $builder;
    }

    /**
     * Gets filters and general query from the request.
     */
    private function getFiltersAndQuery($request): array
    {
        if (is_null($request)) {
            $request = request();
        }

        // get filters as array
        if ($request instanceof Request) {
            $filters = json_decode($request->input('filters', '{}'), true);
            $generalQuery = $request->input('query');
        } else {
            $filters = $request['filters'] ?? [];
            $filters = is_array($filters) ? $filters : json_decode($filters, true);
            $generalQuery = $request['query'] ?? null;
        }

        if (! isset($filters['filterSet'])) {
            $filters['filterSet'] = [];
        }

        return [
            'filters' => $filters,
            'query' => $generalQuery,
        ];
    }

    /**
     * Checks if column is filterable.
     */
    private function isFilterableColumn($column): bool
    {
        return in_array($column, $this->filterableColumns($column));
    }

    /**
     * Exclude relation from query if exists on excluded keys.
     *
     * @param  Request|array  $request
     */
    public function scopeAddRelations(Builder $builder, $request = null): Builder
    {
        if (is_null($request)) {
            $request = request();
        }

        // excluded/included keys
        if ($request instanceof Request) {
            $excluded = json_decode($request->input('excluded', '[]'), true);
            $included = json_decode($request->input('included', '[]'), true);
        } else {
            $excluded = $request['excluded'] ?? [];
            $included = $request['included'] ?? [];
        }

        $requiredRelations = [];
        $requiredCountRelations = [];

        // add included relations
        if (empty($included)) {
            $requiredRelations = DataTableConfiguration::getRequiredRelations(get_class($builder->getModel()));
            $requiredCountRelations = DataTableConfiguration::getRequiredCountRelations(get_class($builder->getModel()));
        } else {
            foreach ($included as $key) {
                $requiredRelations = array_merge($requiredRelations, Arr::wrap(DataTableConfiguration::getRequiredRelations(get_class($builder->getModel()), explode('.', $key)[0])));
                $requiredCountRelations = array_merge($requiredCountRelations, Arr::wrap(DataTableConfiguration::getRequiredCountRelations(get_class($builder->getModel()), explode('.', $key)[0])));
            }
        }

        // remove excluded relations
        if (! empty($excluded)) {
            foreach ($excluded as $key) {
                if (count(explode('.', $key)) == 1) {
                    $requiredRelations = Arr::except($requiredRelations, DataTableConfiguration::getRequiredRelations(get_class($builder->getModel()), $key));
                    $requiredCountRelations = Arr::except($requiredCountRelations, DataTableConfiguration::getRequiredCountRelations(get_class($builder->getModel()), $key));
                }
            }
        }
        $requiredRelations = array_merge($requiredRelations, array_values(DataTableConfiguration::getPersistentRelations(get_class($builder->getModel()))));
        // builder with required relations
        $builder->with(array_unique($requiredRelations));
        $builder->withCount($requiredCountRelations);

        return $builder;
    }

    /**
     * Add custom columns to query.
     */
    public function scopeAddCustomColumns(Builder $builder): Builder
    {
        if (method_exists($this, 'customColumns')) {
            if (is_null($builder->getQuery()->columns)) {
                $builder->getQuery()->select([$builder->getQuery()->from.'.*']);
            }

            $customColumns = $this->customColumns($builder);

            foreach ($customColumns as $customColumn => $query) {
                $builder->selectSub($query, $customColumn);
            }
        }

        return $builder;
    }

    /**
     * Scope to filter an individual key.
     *
     * @param  null  $value
     * @param  string  $conjunction <and> or <or>
     * @return HasFilters|Builder
     *
     * @throws Exception
     */
    public function scopeFilterKey(Builder $builder, $key, $operator, $value = null, string $conjunction = 'and')
    {
        return $this->filterKey($builder, $key, $operator, $value, $conjunction);
    }

    /**
     * Main function to filter a key.
     *
     * @throws Exception
     */
    public function filterKey(Builder $builder, $key, string $operator, $value = null, string $conjunction = 'and', bool $dotInKey = false): Builder
    {
        // check is a relational key?
        if (is_array($key)) {
            $relation = $key['is_relation'] ? $this->isRelationByDot($key['key']) : false;
            if ($relation) {
                $relation['base'] = $key['base'] ?? true;
            }
            if (isset($key['is_calculated_column']) && $key['is_calculated_column']) {
                $relation = $key;
                if (method_exists($this, 'scope'.$this->getScopeRelationName($relation, $operator))) {
                    // @header('X-filter-handled-by-scope-method: scope'.$this->getScopeRelationName($relation, $operator));
                    $function = $this->getScopeRelationName($relation, $operator);
                    if (($result = $builder->$function($relation, $operator, $value, $conjunction)) !== false) {
                        return $result;
                    }
                } else {
                    // @header('X-filter-NOT-handled-by-scope-method: scope'.$this->getScopeRelationName($relation, $operator));
                }
            }
            $key = $key['key'];
        } elseif (! $dotInKey) {
            $relation = $this->isRelationByDot($key);
        } else {
            $relation = false;
        }

        if ($relation) {
            $relationResult = $this->filterWhereRelation($builder, $relation, $operator, $value, $conjunction);

            if (is_string($relationResult)) {
                $key = $relationResult;
            } else {
                return $relationResult;
            }
        }

        // check calculated columns to filter by having
        if (method_exists($this, 'calculatedColumns') && in_array($key, $this->calculatedColumns())) {
            return $this->filterHaving($builder, $key, $operator, $value, $conjunction);
        }

        if (is_array($value) && $operator == 'contains') {
            $operator = '$';
        }

        // apply the default function of this operator
        $function = $this->operators($operator);

        // Qualify the key to filter by if it is not callable and if it is not an expression
        if (! is_callable($key) && ! $key instanceof Expression) {
            $key = $this->qualifyColumn($key);
        }

        return $this->$function($builder, $key, $value, $conjunction);
    }

    /*
    |--------------------------------------------------------------------------
    | Where Functions
    |--------------------------------------------------------------------------
    |
    */

    /**
     * Where equal|is null|equal date.
     *
     *
     * @return HasFilters|Builder|\Illuminate\Database\Query\Builder
     */
    public function filterWhere(Builder $builder, $key, $value, string $conjunction = 'and')
    {
        if (is_null($value)) {
            if ($this->isNumericCastable($key)) {
                return $builder->where(function (Builder $builder) use ($key) {
                    $builder->where($key, 0)->orWhereNull($key);
                }, null, null, $conjunction);
            } else {
                return $builder->where(function (Builder $builder) use ($key) {
                   $builder->where($key, '')->orWhereNull($key);
                });
            }
        }

        // Date
        if ($this->isDateAttribute($key)) {
            return $this->filterWhereDate($builder, $key, '=', $value, $conjunction);
        }

        return $builder->where($key, '=', $value, $conjunction);
    }

    /**
     * Where not equal|not null|not equal date.
     *
     *
     * @return HasFilters|Builder|\Illuminate\Database\Query\Builder
     */
    public function filterWhereNot(Builder $builder, $key, $value, string $conjunction = 'and')
    {
        if (is_null($value)) {
            if ($this->isNumericCastable($key)) {
                return $builder->where(function (Builder $builder) use ($key) {
                    $builder->where($key, '!=', 0)->whereNotNull($key);
                }, null, null, $conjunction);
            } else {
                $fn = $conjunction == 'and' ? 'where' : 'orWhere';

                return $builder->$fn(function (Builder $builder) use ($key) {
                    $builder->whereNotNull($key)->where($key, '!=', '');
                });
            }
        }

        // Date
        if ($this->isDateAttribute($key)) {
            return $this->filterWhereDate($builder, $key, '!=', $value, $conjunction);
        }

        return $builder->where($key, '!=', $value, $conjunction);
    }

    /**
     * Where Greater than or equal for numeric or date.
     *
     *
     * @return HasFilters|Builder|\Illuminate\Database\Query\Builder
     */
    public function filterWhereGreaterOrEqual(Builder $builder, $key, $value, string $conjunction = 'and')
    {
        // Date
        if ($this->isDateAttribute($key)) {
            return $this->filterWhereDate($builder, $key, '>=', $value, $conjunction);
        }

        return $builder->where($key, '>=', $value, $conjunction);
    }

    /**
     * Where Greater than for numeric or date.
     *
     *
     * @return Builder|\Illuminate\Database\Query\Builder
     */
    public function filterWhereGreater(Builder $builder, $key, $value, string $conjunction = 'and')
    {
        if ($this->isDateAttribute($key)) {
            return $this->filterWhereDate($builder, $key, '>', $value, $conjunction);
        }

        return $builder->where($key, '>', $value, $conjunction);
    }

    /**
     * Where less than or equal for numeric or date.
     *
     *
     * @return Builder|\Illuminate\Database\Query\Builder
     */
    public function filterWhereLessOrEqual(Builder $builder, $key, $value, string $conjunction = 'and')
    {
        if ($this->isDateAttribute($key)) {
            return $this->filterWhereDate($builder, $key, '<=', $value, $conjunction);
        }

        return $builder->where($key, '<=', $value, $conjunction);
    }

    /**
     * Where less than for numeric or date.
     *
     *
     * @return Builder|\Illuminate\Database\Query\Builder
     */
    public function filterWhereLess(Builder $builder, $key, $value, string $conjunction = 'and')
    {
        if ($this->isDateAttribute($key)) {
            return $this->filterWhereDate($builder, $key, '<', $value, $conjunction);
        }

        return $builder->where($key, '<', $value, $conjunction);
    }

    /**
     * Where any value In array.
     */
    public function filterWhereIn(Builder $builder, $key, $values, string $conjunction = 'and'): Builder
    {
        if (empty($values)) {
            return $builder;
        }

        if (! is_array($values)) {
            $values = [$values];
        }

        return $builder->whereIn($key, $values, $conjunction);
    }

    /**
     * Where all values Not In array.
     *
     *
     * @return Builder|\Illuminate\Database\Query\Builder
     */
    public function filterWhereAllNotIn(Builder $builder, $key, $values, string $conjunction = 'and')
    {
        if (empty($values)) {
            return $builder;
        }

        if (is_string($values)) {
            $values = [$values];
        }

        return $builder->whereNotIn($key, $values, $conjunction);
    }

    /**
     * Where all values in array.
     */
    public function filterWhereAllIn(Builder $builder, $key, $values, string $conjunction = 'and'): Builder
    {
        if (empty($values)) {
            return $builder;
        }

        if (is_string($values)) {
            $values = [$values];
        }

        return $builder->select('id')
            ->whereIn($key, $values, $conjunction)
            ->groupBy('id')
            ->havingRaw('COUNT(*) = '.count($values));
    }

    /**
     * Where contains like %value%.
     */
    public function filterWhereContains(Builder $builder, $key, $value, string $conjunction = 'and'): Builder
    {
        return $builder->where($key, 'like', '%'.$value.'%', $conjunction);
    }

    /**
     * Where Not like $value%.
     */
    public function filterWhereNotContains(Builder $builder, $key, $value, string $conjunction = 'and'): Builder
    {
        return $builder->where($key, 'not like', '%'.$value.'%', $conjunction);
    }

    /**
     * Where between two dates.
     */
    public function filterWhereDateBetween(Builder $builder, $key, $value, string $conjunction = 'and'): Builder
    {
        if (empty($value['mode']) && (empty($value['from']) || empty($value['to']))) {
            return $builder;
        }

        if (($mode = $value['mode'] ?? null)) {
            $value = $this->getDateFromMode($value['mode'], $value['numberOfDays'] ?? $value['numberOfMinutes'] ?? 1);
        } else {
            $timezone = Helpers::setting(Setting::KEY_DEFAULT_TIMEZONE, config('app.timezone'));
            $value = ['from' => Carbon::parse($value['from'], $timezone), 'to' => Carbon::parse($value['to'], $timezone)];
        }

        if ($mode == 'pastNumberOfMinutes') {
            $value['from'] = $value['from']->startOfMinute();
        } else {
            $value['from'] = $value['from']->startOfDay();
            $value['to'] = $value['to']->endOfDay();
        }

        return $builder->where(function (Builder $builder) use ($key, $value) {
            return $builder->where($key, '>=', $value['from']->timezone(config('app.timezone')))
                ->where($key, '<=', $value['to']->timezone(config('app.timezone')));
        }, null, null, $conjunction);
    }

    public function filterWhereDate(Builder $builder, $key, $sqlOperator, $value, string $conjunction = 'and'): Builder|\Illuminate\Database\Query\Builder
    {
        if (empty($value)) {
            return $builder;
        }
        if (is_array($value)) {
            $value = $this->getDateFromMode($value['mode'], $value['numberOfDays'] ?? 0);
        } else {
            $timezone = Helpers::setting(Setting::KEY_DEFAULT_TIMEZONE, config('app.timezone'));
            $value = ($value instanceof Carbon) ? $value : Carbon::parse($value, $timezone);
        }

        if ($sqlOperator === '>' || $sqlOperator == '<=') {
            $value = $value->endOfDay()->timezone(config('app.timezone'));
        } else {
            if (is_array($value)) {
                return $builder->where(function (Builder $builder) use ($key, $value) {
                    $builder->where($key, '>=', $value['from']->timezone(config('app.timezone')));
                    $builder->where($key, '<', $value['to']->timezone(config('app.timezone')));
                });
            }
            $value = $value->startOfDay()->timezone(config('app.timezone'));
        }

        if ($sqlOperator === '=') {
            return $builder->{$conjunction == 'and' ? 'where' : 'orWhere'}(function (Builder $builder) use ($key, $value) {
                $builder->where($key, '>=', $value);
                $builder->where($key, '<', $value->clone()->addDay());
            });
        }

        if ($sqlOperator === '!=') {
            return $builder->{$conjunction == 'and' ? 'where' : 'orWhere'}(function (Builder $builder) use ($key, $value) {
                $builder->where($key, '<', $value);
                $builder->orWhere($key, '>=', $value->clone()->addDay());
            });
        }

        return $builder->where($key, $sqlOperator, $value, $conjunction);
    }

    /**
     * Where Starts with a value.
     */
    public function filterWhereStartsWith(Builder $builder, $key, $value, string $conjunction = 'and'): Builder
    {
        return $builder->where($key, 'like', $value.'%', $conjunction);
    }

    /**
     * Where Ends with a value.
     */
    public function filterWhereEndsWith(Builder $builder, $key, $value, string $conjunction = 'and'): Builder
    {
        return $builder->where($key, 'like', '%'.$value, $conjunction);
    }

    /*
    |--------------------------------------------------------------------------
    | Where Relation Functions
    |--------------------------------------------------------------------------
    |
    */

    /**
     * @param  null  $value
     * @return Builder|string <string> if the value is empty and the relation is BelongsTo(one to many)
     *
     * @throws Exception
     */
    public function filterWhereRelation(Builder $builder, array $relation, string $operator, $value = null, string $conjunction = 'and')
    {
        // @header("X-filterWhereRelation: ".json_encode($relation));

        /**
         * check if model define a scope function to this operator on this key?
         *
         * (scope<the function's operator><relation name><relation key>)
         * for example: scopeWhereAllInTags(Builder $builder, $relation, $values, $conjunction = 'and')
         *
         * if the function defined we will overwrite the default function
         */
        if (! $relation['direct_relation'] and ! method_exists($this, 'scope'.$this->getScopeRelationName($relation, $operator))) {
            $relationSplit = explode('.', $relation['combined_key']);
            $relationKey = implode('.', array_slice($relationSplit, 0, -1));
            if($operator == 'isEmpty') {
                return $this->builder->doesntHave($relationKey);
            }
            return $builder->has($relationKey, '>=', 1, $conjunction, function (Builder $builder) use ($value, $operator, $relationSplit) {
                    return $builder->filterKey($relationSplit[count($relationSplit) - 1], $operator, $value);
            })->when($operator == '!=', function (Builder $builder) use($relationKey) {
                $builder->orDoesntHave($relationKey);
            });

            // Dead code (never reached):
            throw new Exception('The Filter(HasFilters trait) only support direct relations. You can make your query as custom scope.');
        }

        if (method_exists($this, 'scope'.$this->getScopeRelationName($relation, $operator))) {
            // @header('X-filter-handled-by-scope-method: scope'.$this->getScopeRelationName($relation, $operator));
            $function = $this->getScopeRelationName($relation, $operator);
            if (($result = $builder->$function($relation, $operator, $value, $conjunction)) !== false) {
                return $result;
            }
        } else {
            // @header('X-filter-NOT-handled-by-scope-method: scope'.$this->getScopeRelationName($relation, $operator));
        }

        if ($value === 0 || $value === false || (! empty($value) && ! is_null($value))) {
            // has use simi-joins(exists, in) instead of joins, which is more efficient in our case
            // https://blog.jooq.org/2016/03/09/sql-join-or-exists-chances-are-youre-doing-it-wrong/
            // If you need to check whether you have any matches between a table A and a table B,
            // but you only really care about the results from table A, do make sure you're using a
            // SEMI-JOIN (i.e. an EXISTS or IN predicate), not an (INNER) JOIN.  must the model of
            // this relation implement from Filterable interface and used HasFilters trait

            // get instance from relation
            $relationInstance = $builder->getRelation($relation['name']);

            if ($relation['direct_relation']) {
                if ($relationInstance instanceof BelongsTo) {
                    return $this->filterWhereBelongsToRelation($builder, $relationInstance, $relation, $operator, $value, $conjunction);
                } elseif ($relationInstance instanceof MorphToMany) {
                    return $this->filterWhereMorphToManyRelation($builder, $relationInstance, $relation, $operator, $value, $conjunction);
                } elseif ($relationInstance instanceof HasMany || $relationInstance instanceof HasOne) {
                    return $this->filterWhereHasManyRelation($builder, $relationInstance, $relation, $operator, $value, $conjunction);
                }
            }

            $relationSplit = explode('.', $relation['combined_key']);

            return $builder->has(implode('.', array_slice($relationSplit, 0, -1)), '>=', 1, $conjunction, function (Builder $builder) use ($value, $operator, $relationSplit) {
                return $builder->filterKey($relationSplit[count($relationSplit) - 1], $operator, $value);
            });
        } else {
            // if empty value and the relation is BelongsTo(one to many)
            if ($this->{$relation['name']}() instanceof BelongsTo) {
                return $this->{$relation['name']}()->getForeignKeyName();
            } else {
                // if empty value and the relation is not BelongsTo(one to many)

                if ($relation['base'] ?? true) {
                    $function = ($operator == '=' || $operator == 'isEmpty') ? 'whereDoesntHave' : 'whereHas';

                    return $builder->$function($relation['name']);
                } else {
                    return $builder->whereHas($relation['name'], function (Builder $builder) use ($relation, $operator) {
                        $function = ($operator == '=' || $operator == 'isEmpty') ? 'whereNull' : 'whereNotNull';

                        $builder->$function($relation['key']);
                    });
                }
            }
        }
    }

    /**
     * Where the relation is MorphToMany.
     *
     * We will use "JOIN" when the operator not requires the "HAVING" SQL statement, otherwise we will use "EXISTS"
     */
    public function filterWhereMorphToManyRelation(Builder $builder, MorphToMany $relationInstance, array $relation, string $operator, $value, string $conjunction = 'and'): Builder
    {
        // @header("X-filterWhereMorphToManyRelation: ".json_encode($relation));
        // check if the operator requires "having" SQL statement
        if ($this->isHavingRequired($operator, $value) || ! $this->isRelationsByJoin()) {
            $function = $conjunction == 'and' ? 'whereHas' : 'orWhereHas';
            if ($operator === '!=') {
                $function = $conjunction == 'and' ? 'whereDoesntHave' : 'orWhereDoesntHave';
                $operator = '=';
            }

            return $builder->$function($relationInstance->getRelationName(), function (Builder $builder) use ($relation, $relationInstance, $operator, $value) {
                $key = ($relation['base'] ?? true) ? $relationInstance->getQualifiedRelatedPivotKeyName() : $relationInstance->qualifyColumn($relation['key']);

                if (is_array($value) && $operator == '=') { // exactly
                    $builder->groupBy($relationInstance->getQualifiedForeignPivotKeyName());

                    sort($value);

                    $builder->havingRaw("GROUP_CONCAT({$this->getKeyToSQL($key)} ORDER BY {$this->getKeyToSQL($key)}) = '".implode(',', $value)."'");
                } else {
                    $builder->filterKey([
                        'key' => $key,
                        'is_relation' => false,
                    ], $operator, $value);
                }
            }, '>=', $operator == '=' || ! is_array($value) ? 1 : count($value));
        }

        $function = $this->operators($operator);

        $builder->leftJoin($relationInstance->getTable(), function (JoinClause $join) use ($relationInstance) {
            $join->on(
                $relationInstance->getParent()->getTable().'.'.$relationInstance->getParentKeyName(),
                '=',
                $relationInstance->getTable().'.'.$relationInstance->getForeignPivotKeyName()
            )
                ->where($relationInstance->getTable().'.'.$relationInstance->getMorphType(), '=', $relationInstance->getMorphClass());
        })
            ->select($relationInstance->getParent()->getTable().'.'.'*')
            ->groupBy($relationInstance->getParent()->getTable().'.'.$relationInstance->getParent()->getKeyName());

        return $this->$function($relationInstance->getTable().'.'.$relationInstance->getRelatedPivotKeyName(), $value, $conjunction);
    }

    /**
     * Where the relation is HasMany.
     *
     * We will use "JOIN" when the operator not requires the "HAVING" SQL statement, otherwise we will use "EXISTS"
     *
     * @param  HasMany|HasOne  $relationInstance
     */
    public function filterWhereHasManyRelation(Builder $builder, $relationInstance, array $relation, string $operator, $value, string $conjunction = 'and'): Builder
    {
        // @header("X-filterWhereHasManyRelation: ".json_encode($relation));
        // check if the operator requires "having" SQL statement
        if ($this->isHavingRequired($operator, $value) || ! $this->isRelationsByJoin()) {
            $function = $conjunction == 'and' ? 'whereHas' : 'orWhereHas';

            return $builder->$function($relation['name'], function (Builder $builder) use ($relationInstance, $operator, $value, $relation) {
                $builder->select($relationInstance->getForeignKeyName())
                    ->groupBy($relationInstance->getForeignKeyName());

                if ($operator == '$') {
                    $builder->filterKey($relation['key'], $operator, $value)
                        ->havingRaw('COUNT(`'.$relationInstance->getForeignKeyName().'`) = '.count($value));
                } elseif ($operator == '=' && is_array($value)) { // exactly
                    sort($value);
                    $builder->havingRaw('GROUP_CONCAT(`'.$relationInstance->getRelated()->getTable().'`.`'.$relation['key']."`) = '".implode(',', $value)."'");
                } else {
                    $builder->filterKey($relation['key'], $operator, $value);
                }
            });
        }

        $function = $this->operators($operator);

        $builder->leftJoin($relationInstance->getRelated()->getTable(), function (JoinClause $join) use ($relationInstance) {
            $join->on(
                $relationInstance->getParent()->getTable().'.'.$relationInstance->getLocalKeyName(),
                '=',
                $relationInstance->getRelated()->getTable().'.'.$relationInstance->getForeignKeyName()
            );
            foreach ($relationInstance->getQuery()->getQuery()->wheres as $where) {
                //        $whereFunction = 'where' . ( $where['type'] == 'Basic' ? '' : ucfirst( $where['type'] ) );
                if ($where['type'] == 'Basic') {
                    $join->where($relationInstance->getRelated()->getTable().'.'.$where['column'], $where['operator'], DB::raw("'".$where['value']."'"), $where['boolean']);
                }
            }
        })
            ->select($relationInstance->getParent()->getTable().'.'.'*')
            ->groupBy($relationInstance->getParent()->getTable().'.'.$relationInstance->getParent()->getKeyName());

        return $this->$function($relationInstance->getRelated()->getTable().'.'.$relation['key'], $value, $conjunction);
    }

    public function filterWhereBelongsToRelation(Builder $builder, BelongsTo $relationInstance, array $relation, string $operator, $value, $conjunction = 'and')
    {
        if ($relationInstance->getOwnerKeyName() == $relation['key']) {
            return $relationInstance->getForeignKeyName();
        }

        if (! $this->isRelationsByJoin()) {
            $function = $conjunction == 'and' ? 'whereHas' : 'orWhereHas';

            return $builder->$function($relation['name'], function (Builder $builder) use ($value, $operator, $relation) {
                //                $function = $conjunction == 'and' ? 'where' : 'orWhere';
                //                $builder->{$function}($relation['key'], $operator, $value);
                /*
                 * The line below caused a bug.  filterKey is not a method on the Builder class.
                 */
                $builder->filterKey($relation['key'], $operator, $value);
            });
        }

        // check if the operator requires "having" SQL statement
        if ($this->isHavingRequired($operator, $value)) {
            return $builder;
        }

        $function = $this->operators($operator);

        $builder->leftJoin($relationInstance->getRelated()->getTable(), function (JoinClause $join) use ($relationInstance) {
            $join->on(
                $relationInstance->getParent()->getTable().'.'.$relationInstance->getForeignKeyName(),
                '=',
                $relationInstance->getRelated()->getTable().'.'.$relationInstance->getOwnerKeyName()
            );
            foreach ($relationInstance->getQuery()->getQuery()->wheres as $where) {
                //        $whereFunction = 'where' . ( $where['type'] == 'Basic' ? '' : ucfirst( $where['type'] ) );
                if ($where['type'] == 'Basic') {
                    $join->where($relationInstance->getRelated()->getTable().'.'.$where['column'], $where['operator'], DB::raw($where['value']), $where['boolean']);
                }
            }
        })->select($relationInstance->getParent()->getTable().'.'.'*')
            ->groupBy($relationInstance->getParent()->getTable().'.'.$relationInstance->getParent()->getKeyName());

        return $this->$function($relationInstance->getRelated()->getTable().'.'.$relation['key'], $value, $conjunction);
    }

    /**
     * Filtering by having.
     */
    public function filterHaving(Builder $builder, string $key, string $operator, $value, string $conjunction = 'and'): Builder
    {
        if ($operator == 'contains' || $operator == 'doesNotContain') {
            $operator = $operator == 'contains' ? 'like' : 'not like';
            $value = "%$value%";
        }

        return $builder->having($key, $operator, $value, $conjunction);
    }

    public function scopeFilterWhereNestedRelation(Builder $builder, array $relation, string $operator, $value, $conjunction)
    {
        $function = $conjunction == 'and' ? 'whereHas' : 'orWhereHas';

        $keyExploded = explode('.', $relation['combined_key']);
        $relation = implode('.', array_slice($keyExploded, 0, count($keyExploded) - 1));
        $lastKey = array_slice($keyExploded, -1)[0];

        return $builder->{$function}($relation, function (Builder $builder) use ($lastKey, $value, $operator) {
            $builder->filterKey([
                'key' => $builder->qualifyColumn($lastKey),
                'is_relation' => false,
            ], $operator, $value);
        });
    }

    /**
     * Get the appropriate date from the mode.
     *
     *
     * @return array|Carbon
     */
    public function getDateFromMode($mode, int $numberOfUnits = 1)
    {
        $timezone = Helpers::setting(Setting::KEY_DEFAULT_TIMEZONE, config('app.timezone'));

        $now = Carbon::now($timezone);

        switch ($mode) {
            case 'today':
                return Carbon::today($timezone);
            case 'tomorrow':
                return Carbon::tomorrow($timezone);
            case 'yesterday':
                return Carbon::yesterday($timezone);
            case 'oneWeekAgo':
                return $now->subWeek();
            case 'oneWeekFromNow':
                return $now->addWeek();
            case 'oneMonthAgo':
                return $now->subMonth();
            case 'oneMonthFromNow':
                return $now->addMonth();
            case 'daysAgo':
                return $now->subDays($numberOfUnits);
            case 'daysFromNow':
                return $now->addDays($numberOfUnits);
            case 'thisWeek': // we added
                return ['from' => $now->clone()->startOfWeek(), 'to' => $now->clone()->endOfWeek()];
            case 'thisMonth': // we added
                return ['from' => $now->clone()->startOfMonth(), 'to' => $now->clone()->endOfMonth()];
            case 'thisYear': // we added
                return ['from' => $now->clone()->startOfYear(), 'to' => $now->clone()->endOfYear()];
            case 'pastWeek':
                return ['from' => $now->clone()->subWeek()->startOfWeek(), 'to' => $now->clone()->subWeek()->endOfWeek()];
            case 'pastMonth':
                return ['from' => $now->clone()->subMonth()->startOfMonth(), 'to' => $now->clone()->subMonth()->endOfMonth()];

            // https://siberventures.atlassian.net/browse/SKU-5785?focusedCommentId=32746
            case 'pastYear':
                //return ['from' => $now->clone()->subYear()->startOfYear(), 'to' => $now->clone()->subYear()->endOfYear()];
                return ['from' => $now->clone()->subYear(), 'to' => $now->clone()];

            case 'nextWeek':
                return ['from' => $now->clone()->addWeek()->startOfWeek(), 'to' => $now->clone()->addWeek()->endOfWeek()];
            case 'nextMonth':
                return ['from' => $now->clone()->addMonth()->startOfMonth(), 'to' => $now->clone()->addMonth()->endOfMonth()];
            case 'nextYear':
                return ['from' => $now->clone()->addYear()->startOfYear(), 'to' => $now->clone()->addYear()->endOfYear()];
            case 'nextNumberOfDays':
                return ['from' => $now, 'to' => $now->clone()->addDays($numberOfUnits)];
            case 'pastNumberOfDays':
                return ['from' => $now->clone()->subDays($numberOfUnits), 'to' => $now];
            case 'pastNumberOfMinutes':
                return ['from' => $now->clone()->subMinutes($numberOfUnits), 'to' => $now];
            default:
                return $now;
        }
    }

    public function isHavingRequired($operator, $value)
    {
        if (is_array($value) and $operator == '=') {
            return true;
        }

        return in_array($operator, ['$']);
    }

    /**
     * Get the corresponded value of operator.
     *
     *
     * @return mixed
     */
    public function valueByOperator(string $operator, $value)
    {
        switch ($operator) {
            case 'isNoneOf':
            case 'isAnyOf':
            case '|':
            case '$':
                if (! is_array($value)) {
                    //          $value = array_filter([$value]);
                    $value = explode(',', $value);
                }
                break;
            case 'isNotEmpty':
            case 'isEmpty':
            case 'isAssigned':
            case 'isNotAssigned':
                $value = null;
                break;
        }

        return $value;
    }

    /**
     * Determine if the given attribute is a date or date castable.
     */
    protected function isDateAttribute($key): bool
    {
        if (is_callable($key) && $key != 'date') {
            return false;
        }

        if ($key instanceof Expression) {
            return false;
        }

        $keySplit = explode('.', $key);

        return parent::isDateAttribute($keySplit[count($keySplit) - 1]);
    }

    /**
     * Determine whether key is numeric.
     */
    protected function isNumericCastable(string $key): bool
    {
        if (is_callable($key)) {
            return false;
        }

        $keySplit = explode('.', $key);

        return $this->hasCast($keySplit[count($keySplit) - 1], [
            'int',
            'integer',
            'real',
            'float',
            'double',
            'decimal',
            'bool',
            'boolean',
        ]);
    }

    private function getKeyToSQL(string $key)
    {
        $sqlKey = [];
        foreach (explode('.', $key) as $item) {
            $sqlKey[] = '`'.$item.'`';
        }

        return implode('.', $sqlKey);
    }

    private function getDataTableClass(Builder $builder)
    {
        if ($builder->getModel()->dataTableClass) {
            return $builder->getModel()->dataTableClass;
        }

        return get_class($builder->getModel());
    }

    /**
     * Is key from relation (check by "." between key).
     *
     *
     * @return array|bool <false> if not relation
     */
    public function isRelationByDot($key)
    {
        if (is_callable($key)) {
            return false;
        }

        $count = count($relation = explode('.', $key));
        if ($count > 1) {
            // the relation name and the key inside the relation
            return [
                'name' => $relation[0],
                'key' => $relation[1],
                'subkey' => isset($relation[2]) ? $relation[2] : null,
                'direct_relation' => $count == 2,
                'combined_key' => $key,
            ];
        }

        return false;
    }

    private function getScopeRelationName(array $relation, string $operator, $for = 'filter')
    {
        $prefix = ($for == 'filter') ? 'Filter' : 'Sort';
        static $count = 0;
        $count++;
        // @header('X-scope-relation-'.$count.': '.$prefix.ucfirst($relation['name']));

        return $prefix.ucfirst($relation['name']);
    }

    private function isRelationsByJoin()
    {
        return property_exists($this, 'relationsByJoin') ? $this->relationsByJoin : false;
    }
}
