<?php

namespace App\Models\Concerns;

use App\Enums\BulkImport\DataImportTypeEnum;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Str;
use Throwable;

/**
 * This needs to be imported directly in the model instead of AbstractClass.
 * We are using new self() here and this will throw an error for abstract class if not imported inside the model.
 */
trait BulkImport
{
    /**
     * This purges database connection after upsert.
     */
    private bool $purgeDatabaseConnection = true;

    private $createTemporaryTables = true;

    private $mappings;

    private $defaultColumns;

    public ?string $temporaryTable;

    private $dataToImport;

    private string $databaseConnectionForCreateAndLAterStatements = 'mysql';

    private $databaseConnection = 'mysql';

    private $importableColumnNames;

    private $allSheetColumnsNames;

    private $allSheetColumnsNamesWithTablePrefix;

    private $importableColumnNamesWithoutPrefix;

    private $importableColumnNamesWithTablePrefix;

    private $uniqueByColumns;

    private $sqlStatements;

    private $sqlStatementsAfterJoin;

    private $whereConditionForUpsert;

    private $sequenceId = 'id';

    private $updateBackup = false;

    private bool $loadOriginalRecordId = false;

    private $columnTypesForMigration = [
        'string' => 'text',
        'text' => 'text',
        'dateTime' => 'dateTime',
        'datetime' => 'datetime',
        'date' => 'date',
        'json' => 'json',
        'decimal' => 'decimal',
        'double' => 'decimal',
        'integer' => 'integer',
        'boolean' => 'boolean',
    ];

    private $castColumnsType = [
        'string' => 'text',
        'text' => 'text',
        'dateTime' => 'timestamp',
        'datetime' => 'timestamp',
        'decimal' => 'decimal',
        'double' => 'decimal',
        'integer' => 'integer',
        'json' => 'json',
        'date' => 'date',
        'boolean' => 'boolean',
    ];

    /**
     * @throws Throwable
     */
    public function scopeDataFeedBulkImport($builder, $args): ?string
    {
        cache()->set('test-implicit-commit', true);
        ini_set('memory_limit', -1);

        if (! is_null(@$args['load_original_record_id'])) {
            $this->loadOriginalRecordId = $args['load_original_record_id'];
        }

        if (isset($args['purgeDatabaseConnection'])) {
            $this->purgeDatabaseConnection = $args['purgeDatabaseConnection'];
        }

        //load mappings in variable
        $this->mappings = collect($args['mappings'])
            ->map(function ($column) {
                if (is_null(@$column['is_importable'])) {
                    $column['is_importable'] = true;
                }

                return $column;
            });

        //Handle join columns
        $joinColumns = collect([]);
        foreach ($this->mappings->whereNotNull('joins') as $mapping) {
            foreach ($mapping['joins'] as $join) {
                if (is_array($join['add_select'])) {
                    foreach ($join['add_select'] as $addSelect) {
                        if (is_null(@$addSelect['is_importable'])) {
                            $addSelect['is_importable'] = true;
                        }

                        $addSelect['is_joined_column'] = true;

                        if ($this->mappings->where('expected_column_name', $addSelect['expected_column_name'])->count() === 0) {
                            $joinColumns->push($addSelect);
                        }
                    }
                }
            }
        }

        //Default columns
        $this->defaultColumns = collect($args['default_columns'])
            ->map(function ($column) {
                if (is_null(@$column['is_importable'])) {
                    $column['is_importable'] = true;
                }

                return $column;
            });

        //Make sure stage columns exist in mappings
        foreach ($this->mappings->whereNotNull('joins') as $mapping) {
            foreach ($mapping['joins'] as $join) {
                foreach ($join['on'] as $joinOn) {
                    if (
                        $this->mappings->where('expected_column_name', $joinOn['stage_column_to_join'])->count() === 0
                        && $this->defaultColumns->where('expected_column_name', $joinOn['stage_column_to_join'])->count() === 0
                    ) {
                        if (is_null(@$joinOn['is_join_column'])) {
                            throw_if(
                                $args['data_to_import']['type'] !== DataImportTypeEnum::DB_QUERY(),
                                new \Exception('Missing mapping for '.$joinOn['stage_column_to_join'], 1)
                            );
                        }
                    }
                }
            }
        }

        $this->mappings = $this->mappings->merge($joinColumns);
        $this->uniqueByColumns = collect($args['unique_by_columns']);

        $this->whereConditionForUpsert = @$args['where_conditions_for_upsert'];

        if (@$args['sequence_id']) {
            $this->sequenceId = $args['sequence_id'];
        }

        if (@$args['sql_statements_before_upsert']) {
            $this->sqlStatements = collect($args['sql_statements_before_upsert']);
        }

        if (@$args['sql_statements_after_joins']) {
            $this->sqlStatementsAfterJoin = collect($args['sql_statements_after_joins']);
        }

        $this->updateBackup = @$args['update_backup'] ? $args['update_backup'] : false;

        //Set data to import
        $this->dataToImport = $args['data_to_import'];

        // if ($this->purgeDatabaseConnection) {
        //     Config::set('database.connections.mysql.strict', false);
        //     DB::purge($this->databaseConnection);
        // }

        //Make select columns
        $this->createTemporaryTableName();
        $this->makeListForImportableColumns();
        $this->makeListForAllColumns();

        //load data in temporary table
        $this->loadDataInTemporaryTable();

        //execute sql statements
        $this->executeSqlStatementsBeforeUpsert($this->sqlStatements);

        //Load joins data
        $this->loadJoinsData();

        //execute sql statements
        $this->executeSqlStatementsBeforeUpsert($this->sqlStatementsAfterJoin);

        //create index
        // $this->addIndexForUniqueColumns();

        //before import hook
        $this->beforeImportHook();

        //Default upsert as true);
        if (is_null(@$args['update']) || @$args['update'] === true) {
            $this->updateDataFeed();
        }

        if (is_null(@$args['insert']) || @$args['insert'] === true) {
            $this->insertDataFeed();
        }

        //after import hook
        $this->afterImportHook();

        if ($this->loadOriginalRecordId) {
            $this->loadOriginalRecordIdInTemporaryTable();
        }

        // if ($this->purgeDatabaseConnection) {
        //     Config::set('database.connections.mysql.strict', true);
        //     DB::purge('mysql');
        // }

        // if ($this->purgeDatabaseConnection) {
        //     app(TemporaryTableRepository::class)->destroy($this->temporaryTable);
        // }

        return $this->temporaryTable;
    }

    private function createTemporaryTableName()
    {
        //create temporary table
        if (! is_null(@$this->dataToImport['table_name'])) {
            $this->temporaryTable = $this->dataToImport['table_name'];
        } else {
            $this->temporaryTable = strtolower('temporary_table_'.preg_replace('~[0-9]~', '', Str::random(12)));
        }
    }

    private function dropColumnFromTemporaryTable($column)
    {
        try {
            DB::connection($this->databaseConnectionForCreateAndLAterStatements)->statement("ALTER TABLE $this->temporaryTable DROP $column");
        } catch (\Exception $e) {
        }
    }

    private function addIncrements()
    {
        $this->dropColumnFromTemporaryTable($this->sequenceId);

        Schema::connection($this->databaseConnectionForCreateAndLAterStatements)
            ->table($this->temporaryTable, function (Blueprint $table) {
                $table->bigIncrements($this->sequenceId);
            });
    }

    private function loadJson()
    {
        $json = $this->dataToImport['data'];

        //Handle single quotes
        $json = str_replace("'", "''", $json);

        //TODO
        if (env('USE_CURL_FOR_BULK_IMPORT', false)) {
        } else {
            //JSON decode to array - This is temporary solution and we need to figure out a way to simply insert JSON directly in MySQL.
            //We can also use jq utility in linux to manipulate JSON if reequired.
            $array = json_decode($json, true);

            //Create temp file

            $delimiter = ';';
            $enclosure = '"';
            $escape_char = '\\';
            $tempFile = tmpfile();

            $f = fopen(stream_get_meta_data($tempFile)['uri'], 'w+');

            foreach ($array as $item) {
                fputcsv($f, [json_encode($item)], $delimiter, $enclosure, $escape_char);
            }

            rewind($f);

            //Create temp table
            Schema::connection($this->databaseConnectionForCreateAndLAterStatements)
                ->create($this->temporaryTable, function (Blueprint $table) {
                    if ($this->createTemporaryTables) {
                        $table->temporary();
                    }

                    $table->jsonb('json_data');
                });

            $query = "LOAD DATA LOCAL INFILE '".stream_get_meta_data($tempFile)['uri']."'
                        INTO TABLE $this->temporaryTable
                        FIELDS TERMINATED BY ';'
                        ENCLOSED BY '\"'
                        ESCAPED BY '\"'
                        LINES TERMINATED BY '\n'";

            DB::connection($this->databaseConnectionForCreateAndLAterStatements)->statement($query);
        }

        //Add ID integer
        $this->addIncrements();
    }

    private function loadQuery()
    {
        DB::connection($this->databaseConnectionForCreateAndLAterStatements)->statement('CREATE '.($this->createTemporaryTables ? 'TEMPORARY' : '')." TABLE $this->temporaryTable AS ".$this->dataToImport['query']->toSqlWithBindings());
        $this->addIncrements();
    }

    private function loadRawQuery()
    {
        DB::connection($this->databaseConnectionForCreateAndLAterStatements)->statement('CREATE '.($this->createTemporaryTables ? 'TEMPORARY' : '')." TABLE $this->temporaryTable AS ".$this->dataToImport['query']);
        $this->addIncrements();
    }

    //TEMP CODE
    private function loadCsv()
    {
        $file = fopen($this->dataToImport['file'], 'r');
        $csv = [];
        $headers = fgetcsv($file);

        foreach ($headers as $key => $header) {
            $headers[$key] = preg_replace('/[\x00-\x1F\x7F-\xFF]/', '', $header);
        }

        while (($result = fgetcsv($file)) !== false) {
            $_result = [];
            foreach ($headers as $key => $header) {
                $_result[$header] = $result[$key];
            }
            $csv[] = $_result;
        }
        $this->dataToImport['data'] = json_encode($csv);
        $this->dataToImport['type'] = 'json';

        $this->loadDataInTemporaryTable();
    }

    private function createMappings()
    {
        $this->mappings->each(function ($mapping) {
            if (@$mapping['data_column_name'] && ! str_contains($mapping['data_column_name'], '*')) {
                $this->dropColumnFromTemporaryTable($mapping['expected_column_name']);

                Schema::connection($this->databaseConnectionForCreateAndLAterStatements)
                    ->table($this->temporaryTable, function (Blueprint $table) use ($mapping) {
                        $columnType = $this->columnTypesForMigration[$mapping['expected_column_type']];
                        if ($columnType === 'decimal') {
                            $table->{$columnType}($mapping['expected_column_name'], 24, 8)->nullable();
                        } else {
                            $table->{$columnType}($mapping['expected_column_name'])->nullable();
                        }
                    });
            }
        });
    }

    private function populateColumns()
    {
        $collectSelectClause = [];

        foreach ($this->mappings as $mapping) {
            if (@$mapping['data_column_name'] && ! str_contains($mapping['data_column_name'], '*')) {
                $selectClauseExtractedFromJson = $this->getSelectSqlForJsonData('json_data', $mapping['data_column_name']);
                $castJsonColumnTo = $this->castColumnsType[$mapping['expected_column_type']];
                $updateValue = "$selectClauseExtractedFromJson";

                //This is used to remove the comma, percent etc.
                switch ($castJsonColumnTo) {
                    case 'integer':
                        $updateValue = "
                        (CASE 
                            WHEN $updateValue != 'null'
                            THEN $updateValue
                            ELSE NULL 
                        END)";
                        break;
                    case 'decimal':
                        break;
                    case 'boolean':
                        $updateValue = "
                            (CASE 
                                WHEN $updateValue = 'true' THEN true
                                WHEN $updateValue = 'false' THEN false
                                ELSE  null
                            END)";
                        break;
                    case 'timestamp':
                        $updateValue = "(CASE 
                                            WHEN $updateValue != 'null' 
                                            THEN STR_TO_DATE($updateValue,'%Y-%m-%d %H:%i:%s')
                                            ELSE NULL
                                        END)";
                        break;
                    default:
                        break;
                }

                $collectSelectClause[$mapping['expected_column_name']] = DB::raw($updateValue);
            }
        }

        if (count($collectSelectClause)) {
            DB::connection($this->databaseConnectionForCreateAndLAterStatements)->table($this->temporaryTable)->update($collectSelectClause);
        }
    }

    private function recreateTableFromQuery($query)
    {
        //Handle multiple queries
        $tempTable = strtolower('temporary_table_'.preg_replace('~[0-9]~', '', Str::random(20)));

        DB::connection($this->databaseConnectionForCreateAndLAterStatements)->statement('CREATE '.($this->createTemporaryTables ? 'TEMPORARY' : '')." TABLE $tempTable AS ".$query);
        DB::connection($this->databaseConnectionForCreateAndLAterStatements)->statement("DROP TABLE $this->temporaryTable");
        DB::connection($this->databaseConnectionForCreateAndLAterStatements)->statement('CREATE '.($this->createTemporaryTables ? 'TEMPORARY' : '')." TABLE $this->temporaryTable AS SELECT * FROM $tempTable");
    }

    private function loadMultiDimensionalColumns()
    {
        $select = collect();
        $select->push($this->temporaryTable.'.*');
        foreach ($this->mappings as $mapping) {
            if (@$mapping['data_column_name'] && str_contains($mapping['data_column_name'], '*')) {
                $selectClause = $this->getSelectSqlForJsonData('json_data', $mapping['data_column_name']);
                $alias = $mapping['expected_column_name'];
                $type = $this->castColumnsType[$mapping['expected_column_type']];

                $this->dropColumnFromTemporaryTable($alias);

                $select->push("($selectClause)::$type AS $alias");
            }
        }

        if (count($select) > 1) {
            $this->recreateTableFromQuery('SELECT '.$select->implode(',').' from '.$this->temporaryTable);
        }
    }

    private function loadDefaultColumns()
    {
        $this->defaultColumns->each(function ($column) {
            $this->dropColumnFromTemporaryTable($column['expected_column_name']);

            Schema::connection($this->databaseConnectionForCreateAndLAterStatements)
                ->table($this->temporaryTable, function (Blueprint $table) use ($column) {
                    if (! is_null($column['default_value'])) {
                        $table->{$column['expected_column_type']}($column['expected_column_name'])->default($column['default_value']);
                    } else {
                        $table->{$column['expected_column_type']}($column['expected_column_name'])->nullable();
                    }
                });
        });
    }

    private function loadJoinsData()
    {
        $query = DB::connection($this->databaseConnectionForCreateAndLAterStatements)->table($this->temporaryTable);
        $importableColumns = clone $this->allSheetColumnsNamesWithTablePrefix;
        $hasJoins = false;

        //Add joins
        foreach ($this->mappings->where('joins', '!=', null) as $mapping) {
            $temporaryColumnName = @$mapping['expected_column_name'];
            $hasJoins = true;

            foreach ($mapping['joins'] as $join) {
                if (is_null(@$join['query'])) {
                    $tableAlias = @$join['table_alias'] ?? $join['table'];

                    $query->{$join['type']}(DB::raw($join['table'].' AS '.$tableAlias), function ($query) use ($join, $tableAlias) {
                        foreach ($join['on'] as $on) {
                            $joinOnTable = $tableAlias.'.';

                            if (str_contains($on['database_column'], '.')) {
                                $joinOnTable = '';
                            }

                            $joinOnColumn = "$this->temporaryTable.".$on['stage_column_to_join'];

                            if (@$on['is_join_column']) {
                                $joinOnColumn = $on['stage_column_to_join'];
                            }

                            $query->on($joinOnColumn, '=', $joinOnTable.$on['database_column']);
                        }
                    });
                } else {
                    $query->{$join['type']}($join['query'], $join['table'], function ($query) use ($join) {
                        foreach ($join['on'] as $on) {
                            $query->on("$this->temporaryTable.".$on['stage_column_to_join'], '=', $join['table'].'.'.$on['database_column']);
                        }
                    });
                }

                $table = @$join['table_alias'] ?? $join['table'];

                if (is_array($join['add_select'])) {
                    foreach ($join['add_select'] as $addSelect) {
                        $key = $importableColumns->search("$this->temporaryTable.".$addSelect['expected_column_name']);
                        $columnToSelectFromTable = $addSelect['data_column_name'];
                        $alias = $addSelect['expected_column_name'];
                        $importableColumns[$key] = "$table.$columnToSelectFromTable as $alias";
                    }
                } else {
                    $key = $importableColumns->search("$this->temporaryTable.$temporaryColumnName");
                    $columnToSelectFromTable = $join['add_select'];
                    $alias = $temporaryColumnName;
                    $importableColumns[$key] = "$table.$columnToSelectFromTable::integer as $alias";
                }
            }
        }

        if ($hasJoins) {
            //Add select
            $selectRaw = $importableColumns->implode(',');
            $query->selectRaw("$selectRaw,$this->temporaryTable.".$this->sequenceId.',json_data');

            //Handle multiple queries
            $this->recreateTableFromQuery($query->toSqlWithBindings());
        }
    }

    private function loadDataInTemporaryTable()
    {
        switch ($this->dataToImport['type']) {
            case 'json':
                $this->loadJson();
                $this->createMappings();
                $this->populateColumns();
                $this->loadMultiDimensionalColumns();
                $this->loadDefaultColumns();
                break;

            case 'table':
                $this->createMappings();
                $this->populateColumns();
                $this->loadMultiDimensionalColumns();
                $this->loadDefaultColumns();
                break;

            case DataImportTypeEnum::DB_QUERY():
                $this->loadQuery();
                $this->loadDefaultColumns();
                break;

            case 'raw_query':
                $this->loadRawQuery();
                $this->loadDefaultColumns();
                break;

            case 'csv':
                $this->loadCsv();
                break;
        }
    }

    public function scopeBeforeImportHook($builder)
    {
    }

    public function scopeAfterImportHook($builder)
    {
    }

    public function makeListForImportableColumns()
    {
        //Start with basic
        $this->importableColumnNames = $this->mappings->where('is_importable', true)->pluck('expected_column_name')->filter();

        //Default columns
        $this->importableColumnNames = $this->importableColumnNames->merge($this->defaultColumns->where('is_importable', true)->pluck('expected_column_name'));

        //Add prefix for table name
        $this->importableColumnNamesWithTablePrefix = (clone $this->importableColumnNames)->map(function ($column) {
            return $this->temporaryTable.'.'.$column;
        });
    }

    public function makeListForAllColumns()
    {
        //Start with basic
        $this->allSheetColumnsNames = $this->mappings->pluck('expected_column_name')->filter();

        //Default columns
        $this->allSheetColumnsNames = $this->allSheetColumnsNames->merge($this->defaultColumns->pluck('expected_column_name'));

        //Add prefix for table name
        $this->allSheetColumnsNamesWithTablePrefix = (clone $this->allSheetColumnsNames)->map(function ($column) {
            return $this->temporaryTable.'.'.$column;
        });
    }

    public function scopeMakeSelectQueryForDataFeed($builder)
    {
        //Add joins
        foreach ($this->mappings->where('joins', '!=', null) as $mapping) {
            $temporaryColumnName = $mapping['expected_column_name'];

            foreach ($mapping['joins'] as $join) {
                $tableAlias = @$join['table_alias'] ?? $join['table'];

                $builder->{$join['type']}($tableAlias, function ($query) use ($join, $temporaryColumnName, $tableAlias) {
                    foreach ($join['on'] as $on) {
                        $query->on("$this->temporaryTable.$temporaryColumnName", '=', $tableAlias.'.'.$on['database_column']);
                    }
                });
            }
        }

        //Add select
        $builder->select($this->importableColumnNamesWithTablePrefix->toArray());
    }

    public function scopeAddDistinctClause($builder)
    {
        if ($this->uniqueByColumns && $this->uniqueByColumn->count() > 0) {
            $builder->addSelect(DB::raw('DISTINCT ON ('.$this->uniqueByColumns->implode(',').')'));
        }

        return $builder;
    }

    public function scopeBulkImportFilter($builder, $temporaryTableName, $whereClause)
    {
        $newObject = (new self());

        if ($this->uniqueByColumns->count() > 0) {
            $builder->{$whereClause}(function ($query) use ($temporaryTableName, $newObject) {
                $query->selectRaw('NULL')
                    ->from($newObject->getTable())
                    ->where(function ($query) use ($temporaryTableName, $newObject) {
                        foreach ($this->uniqueByColumns as $uniqueBy) {
                            $query->whereRaw($newObject->getTable().'.'.$uniqueBy." = $temporaryTableName.$uniqueBy");
                        }
                    });
            });
        }

        return $builder->addWhereConditionWhileImporting();
    }

    public function scopeAddWhereConditionWhileImporting($builder)
    {
        foreach ($this->uniqueByColumns as $uniqueColumn) {
            $builder->whereNotNull($uniqueColumn);
        }

        return $builder;
    }

    public function scopeAddSelectClause($builder, $includeAllStageColumns = false)
    {
        $selectColumns = clone $this->importableColumnNamesWithTablePrefix;

        if ($this->uniqueByColumns->count() > 0) {
            if ($includeAllStageColumns) {
                $builder->select(DB::raw($this->allSheetColumnsNamesWithTablePrefix->implode(',')))->groupByRaw($this->uniqueByColumns->implode(','));
            } else {
                $builder->select(DB::raw($this->importableColumnNamesWithTablePrefix->implode(',')))->groupByRaw($this->uniqueByColumns->implode(','));
            }
        }

        return $builder;
    }

    public function getQueryForUpsert($clauseForOperation, $includeAllStageColumns = false)
    {
        $subQuery = (clone $this)
            ->setTable($this->temporaryTable)
            ->addSelectClause($includeAllStageColumns)
            ->bulkImportFilter($this->temporaryTable, $clauseForOperation);

        if ($this->whereConditionForUpsert) {
            foreach ($this->whereConditionForUpsert as $whereCondition) {
                $subQuery = $subQuery->whereRaw($whereCondition);
            }
        }

        $mainQuery = DB::connection($this->databaseConnection)->table(DB::raw("({$subQuery->toSqlWithBindings()}) as sub"))->select(DB::raw($includeAllStageColumns === true ? $this->allSheetColumnsNames->implode(',') : $this->importableColumnNames->implode(',')));

        return $mainQuery;
    }

    private function insertDataFeed()
    {
        $queryForInsert = $this->getQueryForUpsert('whereNotExists');

        DB::connection($this->databaseConnectionForCreateAndLAterStatements)
            ->table($this->getTable())
            ->insertUsing(
                $this->importableColumnNames->toArray(),
                $queryForInsert
            );
    }

    private function updateDataFeed()
    {
        $productionTableAlias = (new self())->getTable();
        $queryForUpdate = $this->getQueryForUpsert('whereExists', true)->toSqlWithBindings();

        $makeSelectForUpdate = (clone $this->importableColumnNames)->map(function ($select) use ($productionTableAlias) {
            return "$productionTableAlias.$select = stage.$select";
        })
            ->implode(',');

        $whereConditionForUpdate = (clone $this->uniqueByColumns)->map(function ($uniqueColumn) {
            return $this->getTable().'.'.$uniqueColumn." = stage.$uniqueColumn";
        })
            ->implode(' AND ');

        if ($this->updateBackup) {
            DB::connection($this->databaseConnectionForCreateAndLAterStatements)->statement('CREATE '.($this->createTemporaryTables ? 'TEMPORARY' : '')." TABLE $this->temporaryTable"."_update_backup AS $queryForUpdate");
        }

        DB::connection($this->databaseConnectionForCreateAndLAterStatements)
            ->statement('UPDATE '.$this->getTable()." as $productionTableAlias, ($queryForUpdate) as stage SET $makeSelectForUpdate WHERE $whereConditionForUpdate");
    }

    public function addIndexForUniqueColumns()
    {
        if ($this->temporaryTable) {
            if ($this->uniqueByColumns->count() > 0) {
                Schema::connection($this->databaseConnectionForCreateAndLAterStatements)
                    ->table($this->temporaryTable, function ($table) {
                        foreach ($this->uniqueByColumns as $uniqueByColumn) {
                            $table->index([$uniqueByColumn], Str::random(5));
                        }

                        $table->index($this->uniqueByColumns->toArray(), Str::random(5));
                    });
            }
        }
    }

    public function executeSqlStatementsBeforeUpsert($sqlStatements = null)
    {
        if ($sqlStatements) {
            foreach ($sqlStatements as $sql) {
                $sql['statement'] = str_replace('{temporaryTable}', "$this->temporaryTable", $sql['statement']);
                DB::connection($this->databaseConnectionForCreateAndLAterStatements)->statement($sql['statement']);
            }
        }
    }

    private function getSelectSqlForJsonData($jsonColumnName, $extractColumn)
    {
        if (strpos($extractColumn, '*')) {
            $columns = explode('.', $extractColumn);
        } else {
            if (! strpos($extractColumn, '.->.')) {
                $columns = [$extractColumn];
            } else {
                $extractColumn = str_replace('.->.', '.', $extractColumn);
                $columns = explode('.', $extractColumn);
            }
        }

        $count = 0;
        $select = '';

        while ($count !== count($columns)) {
            if ($columns[$count] != '*') {
                if ($count === count($columns) - 1) {
                    $select = $select.'"'.$columns[$count].'"';
                } else {
                    $select = '"'.$columns[$count].'".'.$select;
                }
            } else {
                throw new \Exception('TODO - getSelectSqlForJsonData', 1);
            }
            $count++;
        }

        return "(CASE 
                    WHEN JSON_UNQUOTE(JSON_EXTRACT(json_data,'$.$select')) = 'null' 
                    THEN null 
                    ELSE JSON_UNQUOTE(JSON_EXTRACT(json_data,'$.$select')) 
                    END
                )";
    }

    private function loadOriginalRecordIdInTemporaryTable()
    {
        $tableName = $this->getTable();
        $onClauseForUpdate = (clone $this->uniqueByColumns)->map(function ($uniqueColumn) {
            return $this->getTable().'.'.$uniqueColumn." = t_1.$uniqueColumn";
        })
            ->implode(' AND ');

        $whereClauseForUpdate = (clone $this->uniqueByColumns)->map(function ($uniqueColumn) {
            return $this->temporaryTable.'.'.$uniqueColumn." = stage.$uniqueColumn";
        })
            ->implode(' AND ');
        $selectClauseForUpdate = (clone $this->uniqueByColumns)->map(function ($uniqueColumn) {
            return "t_1.$uniqueColumn";
        })
            ->implode(',');

        DB::connection($this->databaseConnection)->statement("
            UPDATE
                $this->temporaryTable AS $this->temporaryTable,
                (
                    SELECT
                        $tableName.id,
                        $selectClauseForUpdate
                    FROM
                        $this->temporaryTable AS t_1
                    LEFT JOIN 
                        $tableName ON $onClauseForUpdate) AS stage 
                    SET 
                        $this->temporaryTable.id = stage.id
            WHERE
                $whereClauseForUpdate");
    }
}
