<?php

namespace App\Models;

use App\Contracts\HasReference;
use App\Models\Concerns\LogsActivity;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Spatie\Activitylog\LogOptions;

/**
 * Class BackorderQueueRelease.
 *
 * @property-read int $id
 * @property int $backorder_queue_id
 * @property int $purchase_order_line_id
 * @property int $released_quantity
 * @property int $link_id
 * @property string $link_type
 * @property Carbon $created_at
 * @property Carbon|null $updated_at
 * @property PurchaseOrderLine $purchaseOrderLine
 * @property BackorderQueue $backorderQueue
 * @property string|null $reference
 */
class BackorderQueueRelease extends Model
{
    use LogsActivity;
    use HasParentLink;

    /*
    |--------------------------------------------------------------------------
    | Implementers
    |--------------------------------------------------------------------------
    */

    protected $fillable = [
        'link_type',
        'link_id',
        'released_quantity',
        'reference',
    ];

    public static function booted(): void
    {
        parent::booted();
        static::created(function ($release) {
            if (! $release->link()->exists()) {
                return;
            }

            $link = $release->link;
            if ($link && in_array(HasReference::class, class_implements($link::class))) {
                $release->update(['reference' => $link->getReference()]);
            }
        });
    }

    public function getParentSubjectIdForActivityLog(): int
    {
        return $this->backorderQueue->salesOrderLine->salesOrder->id;
    }

    public function getMetadataForActivityLog(): ?array
    {
        return [
            'id' => $this->id,
            'backorder_queue_id' => $this->backorder_queue_id,
            'purchase_order_line_id' => $this->purchase_order_line_id,
            'released_quantity' => $this->released_quantity,
        ];
    }

    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logAll()
            ->logExcept(['updated_at'])
            ->logOnlyDirty()
            ->dontSubmitEmptyLogs();
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function link(): MorphTo
    {
        return $this->morphTo('link');
    }

    public function backorderQueue(): BelongsTo
    {
        return $this->belongsTo(BackorderQueue::class);
    }

    /*
    |--------------------------------------------------------------------------
    | Other
    |--------------------------------------------------------------------------
    */

    public function isPurchaseReceipt(): bool
    {
        return $this->link_type === PurchaseOrderShipmentReceiptLine::class;
    }

    public function reduceReleasedQuantityBy($quantity)
    {
        $this->released_quantity = max(0, $this->released_quantity - $quantity);
        $this->save();

        // Update the cache on the queue
        $this->backorderQueue->released_quantity = max(
            0,
            $this->backorderQueue->released_quantity - $quantity
        );
        $this->backorderQueue->save();
    }

    /**
     * Gets the reference of the link.
     */
    public function getReferenceAttribute(): ?string
    {
        if ($this->link instanceof HasReference) {
            return $this->link->getReference();
        }

        return null;
    }
}
