<?php

namespace App\Models;

use App\Exporters\MapsExportableFields;
use App\Importers\DataImporter;
use App\Importers\DataImporters\AttributeDataImporter;
use App\Importers\ImportableInterface;
use App\Models\Concerns\Archive;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use App\Response;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;

/**
 * Class Attribute.
 *
 *
 * @property int $id
 * @property string $name
 * @property string $type
 * @property int|null $attribute_group_id
 * @property array|null $display
 * @property array|null $validation
 * @property bool $available_for_templates
 * @property Carbon|null $archived_at
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property bool $has_options
 * @property int $sort_order
 * @property-read AttributeGroup|null $attributeGroup
 */
class Attribute extends Model implements Filterable, ImportableInterface, MapsExportableFields, Sortable
{
    use Archive, HasFilters, HasSort;
    use HasFactory;

    /**
     * Attribute types.
     */
    const TYPE_STRING = 'string';

    const TYPE_LONGTEXT = 'longtext';

    const TYPE_DATE = 'date';

    const TYPE_DATETIME = 'datetime';

    const TYPE_NUMERIC = 'numeric';

    const TYPE_INTEGER = 'integer';

    const TYPE_CHECKBOX = 'checkbox';

    const TYPES = [
        self::TYPE_STRING,
        self::TYPE_LONGTEXT,
        self::TYPE_DATE,
        self::TYPE_DATETIME,
        self::TYPE_NUMERIC,
        self::TYPE_INTEGER,
        self::TYPE_CHECKBOX,
    ];

    /**
     * Casting.
     */
    protected $casts = [
        'display' => 'array',
        'validation' => 'array',
        'available_for_templates' => 'bool',
        'archived_at' => 'datetime',
    ];

    /**
     * Default values.
     */
    protected $attributes = [
        'type' => self::TYPE_STRING,
    ];

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'name',
        'type',
        'attribute_group_id',
        'display',
        'validation',
        'available_for_templates',
        'has_options',
        'notes',
    ];

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function values()
    {
        return $this->hasMany(AttributeValue::class);
    }

    public function products()
    {
        return $this->belongsToMany(Product::class, 'product_attributes')
            ->using(ProductAttribute::class);
    }

    public function attributeGroup()
    {
        return $this->belongsTo(AttributeGroup::class);
    }

    public function productAttributes()
    {
        return $this->hasMany(ProductAttribute::class);
    }

    public function getHasOptionsAttribute()
    {
        return boolval($this->display['has_options'] ?? false);
    }

    public function setHasOptionsAttribute(bool $value)
    {
        $this->display = array_merge($this->display ?? [], ['has_options' => $value]);
    }

    public function getSortOrderAttribute()
    {
        return $this->display['sort_order'] ?? 0;
    }

    public function setSortOrderAttribute($value)
    {
        $this->display = array_merge($this->display ?: [], ['sort_order' => $value]);
    }

    public function save(array $options = [])
    {
        if (! $this->attribute_group_id) {
            $this->attribute_group_id = null;
        } // Prevent storing attribute_group_id as 0 instead of null

        return parent::save($options);
    }

    /*
    |--------------------------------------------------------------------------
    | Functions
    |--------------------------------------------------------------------------
    */

    /**
     * {@inheritDoc}
     */
    public function delete()
    {
        if (! ((bool) request('ignore_relations', false)) && ($usage = $this->isUsed())) {
            return $usage;
        }

        $this->loadMissing('values');

        // delete the attribute relations
        $this->values()->delete();
        $this->products()->detach();

        return parent::delete();
    }

    /**
     * Check if the attribute has related relations.
     *
     *
     * @return array|bool <false> if not used, <ARRAY> has relations messages
     */
    public function isUsed(bool $addErrorsToResponse = false)
    {
        $relations = [
            'products',
        ];

        $this->loadCount($relations);

        $related = [];

        foreach ($relations as $relatedRelation) {
            $countKeyName = str_replace('-', '_', Str::kebab($relatedRelation)).'_count';
            if ($this->{$countKeyName}) {
                $relatedName = Str::singular(str_replace('-', ' ', Str::kebab($relatedRelation)));

                $related[$relatedRelation] = trans_choice('messages.currently_used', $this->{$countKeyName}, [
                    'resource' => $relatedName,
                    'model' => 'attribute('.$this->name.')',
                ]);

                if ($addErrorsToResponse) {
                    Response::instance()->addError($related[$relatedRelation], ucfirst(Str::singular($relatedRelation)).Response::CODE_RESOURCE_LINKED, $relatedRelation, ['attribute_id' => $this->id]);
                }
            }
        }

        // is it used in integration instance import mappings
        $mappingCount = DataImportMapping::query()->where('mapping.sku_field', "attributes.{$this->name}")->count();
        if ($mappingCount) {
            $related['import_mappings'] = trans_choice('messages.currently_used', $mappingCount, [
                'resource' => 'integration instance listing mapping',
                'model' => 'attribute('.$this->name.')',
            ]);
        }

        return count($related) ? $related : false;
    }

    /**
     * {@inheritDoc}
     */
    public function availableColumns()
    {
        return config('data_table.attribute.columns');
    }

    /**
     * {@inheritDoc}
     */
    public function filterableColumns(): array
    {
        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return ['name', 'type'];
    }

    /**
     * {@inheritDoc}
     */
    public function sortableColumns()
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }

    public function getImporter(string $filePath): DataImporter
    {
        return new AttributeDataImporter(null, $filePath);
    }

    public static function getExportableFields(): array
    {
        return AttributeDataImporter::getExportableFields();
    }
}
