<?php

namespace App\Models;

use App\Abstractions\UniqueFieldsInterface;
use App\Models\Concerns\HasLink;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Carbon;

/**
 * @property int $id
 * @property int $accounting_transaction_id
 * @property string $type
 * @property int $nominal_code_id
 * @property string $description
 * @property int $quantity
 * @property float $amount
 * @property float $tax_amount
 * @property int $tax_rate_id
 * @property int $link_id
 * @property string $link_type
 * @property-read AccountingTransaction $accountingTransaction
 * @property-read NominalCode $nominalCode
 * @property-read TaxRate $taxRate
 * @property Carbon $updated_at
 * @property Carbon $created_at
 */

/*
 * TODO: possibly not needing a link on AccountingTransaction due to these
 */
class AccountingTransactionLine extends Model implements UniqueFieldsInterface
{
    use HasFactory, HasLink;

    const TYPE_DEBIT = 'debit';

    const TYPE_CREDIT = 'credit';

    const TYPE_SALES_INVOICE_LINE = 'sales_invoice_line';

    const TYPE_SALES_CREDIT_LINE = 'sales_credit_line';

    const TYPE_PURCHASE_ORDER_LINE = 'purchase_order_line';

    const TYPE_PURCHASE_INVOICE_LINE = 'purchase_invoice_line';

    const TYPES = [
        self::TYPE_DEBIT,
        self::TYPE_CREDIT,
        self::TYPE_SALES_INVOICE_LINE,
        self::TYPE_SALES_CREDIT_LINE,
        self::TYPE_PURCHASE_ORDER_LINE,
        self::TYPE_PURCHASE_INVOICE_LINE,
    ];

    const TYPES_JOURNAL = [
        self::TYPE_DEBIT,
        self::TYPE_CREDIT,
    ];

    protected $fillable = [
        'accounting_transaction_id',
        'type',
        'nominal_code_id',
        'description',
        'quantity',
        'amount',
        'tax_amount',
        'tax_rate_id',
        'link_id',
        'link_type',
    ];

    public static function getUniqueFields(): array
    {
        return [
            'accounting_transaction_id',
            'type',
            'link_id',
            'link_type',
        ];
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function accountingTransaction(): BelongsTo
    {
        return $this->belongsTo(AccountingTransaction::class);
    }

    public function link()
    {
        return $this->morphTo('link');
    }

    public function nominalCode(): BelongsTo
    {
        return $this->belongsTo(NominalCode::class);
    }

    public function taxRate(): BelongsTo
    {
        return $this->belongsTo(TaxRate::class);
    }

    public function salesOrderLine()
    {
        return $this->morphOne(SalesOrderLine::class, 'link');
    }

    public function salesOrderFulfillmentLine()
    {
        return $this->morphOne(SalesOrderFulfillmentLine::class, 'link');
    }

    public function inventoryAdjustmentLine()
    {
        return $this->morphOne(InventoryAdjustment::class, 'link');
    }

    public function purchaseInvoiceLine()
    {
        return $this->morphOne(PurchaseInvoiceLine::class, 'link');
    }

    public function purchaseReceiptLine()
    {
        return $this->morphOne(PurchaseOrderShipmentReceiptLine::class, 'link');
    }

    public function customerReturnLine()
    {
        return $this->morphOne(SalesCreditReturnLine::class, 'link');
    }

    public function salesCreditLine()
    {
        return $this->morphOne(SalesCreditLine::class, 'link');
    }

    public function stockTakeLine()
    {
        return $this->morphOne(StockTakeItem::class, 'link');
    }

    /*
    |--------------------------------------------------------------------------
    | Functions
    |--------------------------------------------------------------------------
    */

    public function scopeWithSalesOrderLines(Builder $query, array|int $ids)
    {
        return $query->withLink(SalesOrderLine::class, $ids);
    }

    public function scopeWithSalesCreditLines(Builder $query, array|int $ids)
    {
        return $query->withLink(SalesCreditLine::class, $ids);
    }

    public function scopeWithSalesOrderFulfillmentLines(Builder $query, array|int $ids)
    {
        return $query->withLink(SalesOrderFulfillmentLine::class, $ids);
    }

    public function scopeWithPurchaseOrderLines(Builder $query, array|int $ids)
    {
        return $query->withLink(PurchaseOrderLine::class, $ids);
    }

    public function scopeWithPurchaseInvoiceLines(Builder $query, array|int $ids)
    {
        return $query->withLink(PurchaseInvoiceLine::class, $ids);
    }

    public function scopeWithPurchaseOrderReceiptLines(Builder $query, array|int $ids)
    {
        return $query->withLink(PurchaseOrderShipmentReceiptLine::class, $ids);
    }

    public function scopeWithInventoryAdjustments(Builder $query, array|int $ids)
    {
        return $query->withLink(InventoryAdjustment::class, $ids);
    }

    public function scopeWithCustomerReturnLines(Builder $query, array|int $ids)
    {
        return $query->withLink(SalesCreditReturnLine::class, $ids);
    }

    public function scopeWithStockTakeLines(Builder $query, array|int $ids)
    {
        return $query->withLink(StockTakeItem::class, $ids);
    }

    public function scopeWithWarehouseTransferShipmentLines(Builder $query, array|int $ids)
    {
        return $query->withLink(WarehouseTransferShipmentLine::class, $ids);
    }

    public function scopeWithWarehouseTransferReceiptLines(Builder $query, array|int $ids)
    {
        return $query->withLink(WarehouseTransferShipmentReceiptLine::class, $ids);
    }

    /*
    |--------------------------------------------------------------------------
    | Other
    |--------------------------------------------------------------------------
    */

    public function getAmountWithoutTax(?bool $isAmountIncludingTax)
    {
        return $this->amount;
    }

    public function getAmountWithTax(?bool $isAmountIncludingTax)
    {
        if ($isAmountIncludingTax && $this->quantity) {
            return $this->amount + ($this->tax_amount / $this->quantity);
        }

        return $this->amount;
    }

    public function getTaxAmount(?bool $isAmountIncludingTax)
    {
        return $this->tax_amount;
    }
}
