<?php

namespace App\MockSalesChannels;

use App\Integrations\Channel;
use Carbon\Carbon;
use DTS\eBaySDK\Trading\Enums\AckCodeType;
use DTS\eBaySDK\Trading\Enums\OrderStatusCodeType;
use DTS\eBaySDK\Trading\Types\GetOrdersRequestType;
use DTS\eBaySDK\Trading\Types\GetOrdersResponseType;
use DTS\eBaySDK\Trading\Types\GetSellerListRequestType;
use DTS\eBaySDK\Trading\Types\GetSellerListResponseType;
use DTS\eBaySDK\Trading\Types\PaginationType;
use Generator;
use Illuminate\Contracts\Filesystem\FileNotFoundException;

class Ebay extends Channel
{
    /**
     * Get Orders from sales channel.
     *
     * @param  null  $options
     *
     * @throws FileNotFoundException
     */
    public function getSalesOrders($options = null): Generator
    {
        $ordersList = collect($this->mocks('ebay.orders'));

        /**
         * New GetOrdersRequestType if $options null or array of options.
         */
        $perPage = 10;
        if (empty($options) || is_array($options)) {
            $options = new GetOrdersRequestType($options);
        }
        if (empty($options->OrderStatus)) {
            $options->OrderStatus = OrderStatusCodeType::C_ALL;
        }
        if (empty($options->CreateTimeFrom)) {
            $options->CreateTimeFrom = Carbon::now()->subDays(1)->toDateTime();
        }
        if (empty($options->CreateTimeTo)) {
            $options->CreateTimeTo = Carbon::now()->toDateTime();
        }

        /**
         * Set request pagination with initial page if not set.
         */
        if (! isset($options->Pagination)) {
            $pagination = new PaginationType();
            $pagination->EntriesPerPage = $perPage;

            $options->Pagination = $pagination;
        }

        /**
         * Filter Created Time.
         */
        $ordersList = $ordersList->filter(function ($order, $key) use ($options) {
            return Carbon::createFromTimestamp(strtotime($order['CreatedTime']))->gte($options->CreateTimeFrom);
        });

        $ordersList = $ordersList->filter(function ($order, $key) use ($options) {
            return Carbon::createFromTimestamp(strtotime($order['CreatedTime']))->lte($options->CreateTimeTo);
        });

        if (! empty($options->OrderStatus) && $options->OrderStatus != OrderStatusCodeType::C_ALL) {
            $ordersList = $ordersList->where('OrderStatus', $options->OrderStatus);
        }

        /**
         * Return Orders chunks with limit.
         */
        foreach ($ordersList->chunk($options->Pagination->EntriesPerPage) as $chunk_orders) {
            /**
             * Convert array of orders to eBay GetOrdersResponseType with Ack "Success".
             */
            $getOrdersResponse = new GetOrdersResponseType(
                [
                    'OrderArray' => ['Order' => $chunk_orders->values()->all()],
                    'Ack' => AckCodeType::C_SUCCESS,
                ]
            );

            yield $getOrdersResponse;
        }
    }

    /**
     * Check sales channel credentials.
     *
     * @return mixed
     */
    public function checkCredentials()
    {
        // TODO: Implement checkCredentials() method.
    }

    /**
     * Get Products from sales channel.
     *
     * @param  null  $options
     *
     * @throws FileNotFoundException
     */
    public function getProducts($options = null): Generator
    {
        $productsList = collect($this->mocks('ebay.products'));

        /**
         * New GetSellerListRequestType if $options null or array of options.
         */
        if (empty($options) || is_array($options)) {
            $options = new GetSellerListRequestType($options ?? []);

            $options->EndTimeFrom = Carbon::now()->toDateTime();
            $options->EndTimeTo = Carbon::now()->addDays(120)->toDateTime();
        }

        /**
         * Set request pagination with initial page if not set.
         */
        if (! isset($options->Pagination)) {
            $initial_page = 1;

            $pagination = new PaginationType();
            $pagination->PageNumber = $initial_page;
            $pagination->EntriesPerPage = 10;

            $options->Pagination = $pagination;
        }

        /**
         * Filter End Time.
         */
        $productsList = $productsList->filter(function ($product, $key) use ($options) {
            return Carbon::createFromTimestamp(strtotime($product['ListingDetails']['EndTime']))->gte($options->EndTimeFrom);
        });

        $productsList = $productsList->filter(function ($product, $key) use ($options) {
            return Carbon::createFromTimestamp(strtotime($product['ListingDetails']['EndTime']))->lte($options->EndTimeTo);
        });

        /**
         * Return Orders chunks with limit.
         */
        foreach ($productsList->chunk($options->Pagination->EntriesPerPage) as $chunk_products) {
            /**
             * Convert array of orders to eBay GetOrdersResponseType with Ack "Success".
             */
            $getProductsResponse = new GetSellerListResponseType(
                [
                    'ItemArray' => ['Item' => $chunk_products->values()->all()],
                    'Ack' => AckCodeType::C_SUCCESS,
                ]
            );

            yield $getProductsResponse;
        }
    }
}
