<?php

namespace App\Managers;

use App\Data\UserData;
use App\Mail\AccountSetupMail;
use App\Mail\PasswordUpdatedMail;
use App\Repositories\UserRepository;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;

class UserManager
{
    public function __construct(
        protected UserRepository $userRepository
    ) {
    }

    /**
     * @throws \Throwable
     */
    public function createUserAndSendInvitation(UserData $data): Model
    {
        $user = DB::transaction(function () use ($data) {
            $user = $this->userRepository->save($data);
            $this->sendInvitation($user);

            return $user;
        });

        return $user;
    }

    public function sendInvitation($user): void
    {
        $token = Str::random(60);

        $user->activation_token = $token;
        $user->save();

        $url = url('/setup-account/'.$token);
        Mail::to($user->email)->send(new AccountSetupMail($user, $url));
    }

    public function activateUser($token, $password): ?Model
    {
        $user = $this->userRepository->findBy('activation_token', $token);
        $user->password = Hash::make($password);
        $user->activated_at = now();
        $user->activation_token = null;
        $user->save();

        Mail::to($user)->queue(new PasswordUpdatedMail($user));

        return $user;
    }
}
