<?php

namespace App\Managers;

use App\Abstractions\AbstractRepository;
use App\Data\SalesOrderLineTaxLineData;
use App\Data\SalesOrderLineTaxLinesData;
use App\Models\TaxRate;
use App\Repositories\TaxRateRepository;
use Illuminate\Support\Collection;
use Spatie\LaravelData\DataCollection;

class TaxRateManager
{
    public function __construct(
        private readonly TaxRateRepository $taxRates
    ) {}

    /**
     * @param  DataCollection<array-key, SalesOrderLineTaxLinesData> $data
     */
    public function getOrCreateTaxRatesFromTaxData(DataCollection $data): Collection
    {
        $allTaxRates = $this->taxRates->getAll();
        $existingTaxRates = collect();

        foreach ($data as $taxLinesData) {
            $existingTaxRates->push($this->getOrCreateTaxRate($taxLinesData, $allTaxRates));
        }

        return $existingTaxRates;
    }

    public function combineTaxLinesData(SalesOrderLineTaxLinesData $taxLinesData): ?SalesOrderLineTaxLineData
    {
        if (!$taxLinesData->taxLines) {
            return null;
        }
        $rateNames = collect();
        $combinedRate = 0;

        $taxLinesData->taxLines->each(function (SalesOrderLineTaxLineData $taxLine) use($rateNames, &$combinedRate) {
            $rateNames->add($taxLine->rateName);
            $combinedRate += $taxLine->rate;
        });

        $combinedName = $rateNames->implode(' + ');

        return SalesOrderLineTaxLineData::from([
            'rateName' => $combinedName,
            'rate' => $combinedRate,
        ]);
    }

    private function getOrCreateTaxRate(SalesOrderLineTaxLinesData $taxLinesData, Collection $allTaxRates): TaxRate
    {
        $taxLine = $this->combineTaxLinesData($taxLinesData);

        if ($existingTaxRate = $allTaxRates->where('name', $taxLine->rateName)->where('rate', $taxLine->rate)->first()) {
            return $existingTaxRate;
        } else {
            return $this->taxRates->createTaxRateFromNameAndRate($taxLine->rateName, $taxLine->rate);
        }
    }
}