<?php

namespace App\Managers;

use App\Jobs\DeleteProductInventoryJob;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\ProductInventory;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;

class ProductInventoryManager
{
    protected null|int|array $productIds;

    protected bool $updateInventory;

    protected bool $updateAverageCost;

    protected bool $allow_all_products = false;

    public function __construct($productIds = null, $updateInventory = true, $updateAverageCost = true, bool $allow_all_products = false)
    {
        $this->productIds = Arr::wrap($productIds);
        $this->updateInventory = $updateInventory;
        $this->updateAverageCost = $updateAverageCost;
        $this->allow_all_products = $allow_all_products;
    }

    public function updateProductInventoryAndAvgCost()
    {
        customlog('SKU-5982', 'UpdateProductsInventoryAndAvgCost processing started', $this->productIds);
        $products = Product::with($this->getRequiredRelations());

        // update only this product
        if (! empty($this->productIds)) {
            $products->whereIn('id', $this->productIds);
        }

        if (empty($this->productIds) && ! $this->allow_all_products) {
            return;
        }

        $products->chunk(40, function ($products, $page) {
            /** @var Product $product */
            foreach ($products as $product) {
                // Update Product Average Cost
                if ($this->updateAverageCost) {
                    $product->updateAverageCost();
                    // update stock value if we don't need to update the inventory
                    if (! $this->updateInventory) {
                        ProductInventory::query()->where('product_id', $product->id)->update(['inventory_stock_value' => DB::raw("GREATEST(`products_inventory`.`inventory_available`, 0) * {$product->average_cost}")]);
                    }
                }

                // Update Product Inventory
                if ($this->updateInventory) {
                    // used warehouses
                    $inWarehousesQuantity = $this->inWarehousesQuantity($product);

                    $inWarehouseTransitQuantity = $this->inWarehouseTransitQuantity($product);
                    $inWarehousesReservedQuantity = $this->inWarehouseReservedQuantity($product);

                    $warehouseIds = $inWarehousesQuantity->pluck('warehouse_id')
                        ->merge($inWarehouseTransitQuantity->pluck('warehouse_id'))
                        ->merge($inWarehousesReservedQuantity->pluck('warehouse_id'))
                        ->filter()
                        ->unique();

                    // add records only if product has values
                    if ($warehouseIds->isNotEmpty()) {
                        // total inventory for all warehouses
                        $productInventory = new ProductInventory([
                            'product_id' => $product->id,
                            'warehouse_id' => ProductInventory::$idForTotalWarehouses,
                        ]);

                        $productInventory->inventory_total = $inWarehousesQuantity->sum('quantity');
                        $productInventory->inventory_reserved = $inWarehousesReservedQuantity->sum('quantity');
                        $productInventory->inventory_available = $productInventory->inventory_total - $productInventory->inventory_reserved;
                        $productInventory->inventory_in_transit = $inWarehouseTransitQuantity->sum('quantity');
                        $productInventory->inventory_stock_value = $product->getCumulativeCost(); // $productInventory->inventory_total * $product->average_cost;
                        $productInventory->upsertModel();

                        // product inventory for every warehouse
                        foreach ($warehouseIds as $warehouseId) {
                            $inWarehouseInventory = new ProductInventory([
                                'product_id' => $product->id,
                                'warehouse_id' => $warehouseId,
                            ]);

                            $inWarehouseInventory->inventory_total = $inWarehousesQuantity->firstWhere('warehouse_id', $warehouseId)->quantity ?? 0;
                            $inWarehouseInventory->inventory_reserved = $inWarehousesReservedQuantity->firstWhere('warehouse_id', $warehouseId)->quantity ?? 0;
                            $inWarehouseInventory->inventory_available = $inWarehouseInventory->inventory_total - $inWarehouseInventory->inventory_reserved;
                            $inWarehouseInventory->inventory_in_transit = $inWarehouseTransitQuantity->firstWhere('warehouse_id', $warehouseId)->quantity ?? 0;
                            $inWarehouseInventory->inventory_stock_value = max($inWarehouseInventory->inventory_available, 0) * $product->average_cost;
                            $inWarehouseInventory->upsertModel();
                        }

                        // Remove cache for warehouses without movements. 0 for total warehouses
                        dispatch(new DeleteProductInventoryJob($product, $warehouseIds->add(0)->toArray()))->onQueue('syncInventory');
                    } else {
                        // No warehouse has movements for the product,
                        // we clear the cache for the product.
                        dispatch(new DeleteProductInventoryJob($product))->onQueue('syncInventory');
                    }
                }
            }
        });
        customlog('SKU-5982', 'UpdateProductsInventoryAndAvgCost Complete', $this->productIds);
    }

    private function inWarehousesQuantity(Product $product)
    {
        return $product->inventoryMovements()
            ->where('inventory_status', '!=', InventoryMovement::INVENTORY_STATUS_IN_TRANSIT)
            ->select('product_id', 'warehouse_id', DB::raw('sum(quantity) as quantity'))
            ->groupBy(['warehouse_id', 'product_id'])->get();
    }

    private function inWarehouseAvailableQuantity(Product $product)
    {
        return $product->inventoryMovements()
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
            ->select('product_id', 'warehouse_id', DB::raw('sum(quantity) as quantity'))
            ->groupBy(['warehouse_id', 'product_id'])->get();
    }

    private function inWarehouseReservedQuantity(Product $product)
    {
        return $product->inventoryMovements()
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
            ->where('type', '!=', InventoryMovement::TYPE_TRANSFER)
            ->select('product_id', 'warehouse_id', DB::raw('sum(quantity) as quantity'))
            ->groupBy(['warehouse_id', 'product_id'])->get();
    }

    private function inWarehouseTransitQuantity(Product $product)
    {
        return $product->inventoryMovements()
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_IN_TRANSIT)
            ->select('product_id', 'warehouse_id', DB::raw('sum(quantity) as quantity'))
            ->groupBy(['warehouse_id', 'product_id'])->get();
    }

    /**
     * Determined whether we want to update products inventory or not.
     */
    public function setUpdateInventory(bool $update): static
    {
        $this->updateInventory = $update;

        return $this;
    }

    /**
     * Determined whether we want to update products average cost or not.
     */
    public function setUpdateAverageCost(bool $update): static
    {
        $this->updateAverageCost = $update;

        return $this;
    }

    /**
     * Determined whether we want to allow an update of all produts.
     */
    public function setAllowAllProducts(bool $allow_all_products): static
    {
        $this->allow_all_products = $allow_all_products;

        return $this;
    }

    /**
     * Get incoming quantity to the product from opened purchase order lines.
     */
    private function getIncomingQuantity(Product $product): Collection
    {
        $incomingQuantity = collect();

        foreach ($product->purchaseOrderLinesOpened as $purchaseOrderLine) {
            $incoming = $purchaseOrderLine->unreceived_quantity;

            $incomingQuantity->add([
                'warehouse_id' => $purchaseOrderLine->purchaseOrder->destination_warehouse_id,
                'quantity' => max(0, $incoming),
            ]);
        }

        return $incomingQuantity;
    }

    /**
     * Get required relations based on what is need to update.
     */
    private function getRequiredRelations(): array
    {
        if ($this->updateInventory) {
            return [
                'inWarehousesQuantity',
                'inWarehousesReservedQuantity',
                'purchaseOrderLinesOpened',
                'inWarehouseTransitQuantity',
                'activeFifoLayers',
            ];
        }

        if ($this->updateAverageCost) {
            return ['activeFifoLayers', 'fifoLayers'];
        }

        return [];
    }
}
