<?php

namespace App\Managers;

use App\Data\CreateDropshipData;
use App\Data\CreateDropshipLinesData;
use App\Data\CreateDropshipPurchaseOrderData;
use App\Data\CreateDropshipShipmentData;
use App\Data\SalesOrderFulfillmentData;
use App\DTO\PurchaseOrderLineDto;
use App\Exceptions\SalesOrder\SalesOrderFulfillmentDispatchException;
use App\Models\Product;
use App\Models\PurchaseOrder;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\Warehouse;
use App\Repositories\OrderLinkRepository;
use App\Repositories\ProductRepository;
use App\Services\PurchaseOrder\PurchaseOrderManager;
use App\Services\SalesOrder\Fulfillments\FulfillmentManager;
use Illuminate\Contracts\Container\BindingResolutionException;
use Throwable;

class DropshipManager
{
    public function __construct(
        private readonly FulfillmentManager $fulfillmentManager,
        private readonly PurchaseOrderManager $purchaseOrderManager,
        private readonly ProductRepository $products,
        private readonly OrderLinkRepository $links,
    )
    {
    }

    public function create(CreateDropshipData $data): PurchaseOrder
    {
        $salesOrder = SalesOrder::findOrFail($data->sales_order_id);
        $warehouse = Warehouse::findOrFail($data->warehouse_id);

        $purchaseOrderLines = $data->sales_order_lines->map(function(CreateDropshipLinesData $line) use ($salesOrder, $warehouse) {
            $salesOrderLine = $salesOrder->salesOrderLines->where('id', $line->id)->first();
            $product = Product::findOrFail($salesOrderLine->product_id);
            $amount = $this->products->getCostForSupplier($product, $warehouse->id, $warehouse->supplier_id);
            return PurchaseOrderLineDto::from([
                'description' => $salesOrderLine->description,
                'product_id' => $salesOrderLine->product_id,
                'quantity' => $salesOrderLine->quantity,
                'amount' => $amount,
            ]);
        })->toArray();

        $purchaseOrder = $this->purchaseOrderManager->createPurchaseOrder(CreateDropshipPurchaseOrderData::from([
            'sales_order_id' => $salesOrder->id,
            'purchase_order_date' => $salesOrder->order_date, // TODO: Does this make the most logical sense?
            'requested_shipping_method_id' => $data->requested_shipping_method_id,
            'requested_shipping_method' => $data->requested_shipping_method,
            'supplier_id' => $warehouse->supplier_id,
            'supplier_warehouse_id' => $warehouse->id,
            'destination_address_id' => $salesOrder->shipping_address_id,
            'currency_id' => $salesOrder->currency_id,
            'supplier_notes' => $data->supplier_notes,
            'internal_notes' => $data->internal_notes,
            'store_id' => $salesOrder->store_id,
            'purchaseOrderLines' => $purchaseOrderLines,
        ]));

        $this->links->createDropshipOrderLink($salesOrder, $purchaseOrder);
        $purchaseOrder->submit(true, true);
        $salesOrder->updateFulfillmentStatus();

        return $purchaseOrder->refresh();
    }

    /**
     * @throws SalesOrderFulfillmentDispatchException
     * @throws Throwable
     * @throws BindingResolutionException
     */
    public function ship(CreateDropshipShipmentData $data, PurchaseOrder $purchaseOrder): SalesOrderFulfillment
    {
        $salesOrder = $purchaseOrder->salesOrder;

        return $this->fulfillmentManager->fulfill(
            $salesOrder,
            SalesOrderFulfillmentData::from([
                'sales_order_id' => $salesOrder->id,
                'fulfillment_sequence' => 1,
                'warehouse_id' => $purchaseOrder->supplier_warehouse_id,
                'status' => SalesOrderFulfillment::STATUS_FULFILLED,
                'fulfillment_type' => SalesOrderFulfillment::TYPE_MANUAL,
                'fulfilled_at' => now(),
                'user_id' => auth()->id(),
                'fulfilled_shipping_method_id' => $purchaseOrder->requested_shipping_method_id,
                'tracking_number' => $data->tracking_number,
                'fulfillment_lines' => $purchaseOrder->salesOrder->salesOrderLines->map(fn($line) => [
                    'sales_order_line_id' => $line->id,
                    'quantity' => $line->quantity,
                    'fulfillable_quantity' => $line->quantity,
                    'is_dropship' => true,
                ])->toArray(),
            ])->toArray(),
            false,
        );
    }
}