<?php

namespace App\Managers;

use App\DTO\DataImporterDto;
use App\Exceptions\DataImportNoRecordsImportedException;
use App\Exceptions\DataImportTooLargeException;
use App\Exceptions\DataImportValidationErrorsException;
use App\Exceptions\ImportFailedException;
use App\Importers\DataImporter;
use App\Importers\ImportableInterface;

class DataImportManager
{
    public DataImporter $importer;

    public function __construct(ImportableInterface $model, protected DataImporterDto $dataImporterDto)
    {
        $this->setImporter($model);

        if ($this->dataImporterDto->user) {
            $this->importer->setUser($this->dataImporterDto->user);
        }

        // Make sure there are no differences between
        // the uploaded file and the expected fields.
        $this->importer->setRewrites($this->dataImporterDto->parsers);
    }

    private function setImporter(ImportableInterface $model): void
    {
        $this->importer = $model->getImporter(self::getImportFilePath($this->dataImporterDto->stored_name));
        $this->importer->setTask($this->dataImporterDto->task_id);
        $this->importer->setMapping($this->dataImporterDto->mapping);
        $this->importer->setMeta($this->dataImporterDto->meta);

    }

    private function getImportFilePath(string $storedName): string
    {
        $info = config('uploader.models', ['target' => 'imports/models']);

        return storage_path(rtrim($info['target'], '/').'/'.$storedName);
    }

    /**
     * @throws ImportFailedException
     */
    public function getFieldDifferences(): array
    {
        return array_diff(array_map(function ($field) {
            return $field['key'];
        }, $this->importer->importableFields()), $this->importer->getHeaders());
    }

    /**
     * @throws ImportFailedException
     */
    public function getRequiredFieldDifferences(): array
    {
        return array_filter($this->getFieldDifferences(), function ($field) {
            return in_array($field, $this->importer->getRequiredColumns());
        });
    }

    /**
     * @throws ImportFailedException
     */
    public function getPreviewResponse(array $unimportableFields): array
    {
        return [
            'records' => $this->importer->getRecords(),
            'file_headers' => $this->importer->getHeaders(),
            'importable_fields' => $this->importer->importableFields(),
            'difference' => array_values($this->getFieldDifferences()),
            'required_difference' => array_values($this->getRequiredFieldDifferences()),
            'not_importable' => $unimportableFields,
            'task_id' => $this->importer->getTaskId(),
        ];
    }

    private function getRecordsCount(): int
    {
        return $this->importer->getRecordsCount();
    }

    /**
     * @throws DataImportTooLargeException
     * @throws DataImportNoRecordsImportedException
     * @throws DataImportValidationErrorsException
     * @throws ImportFailedException
     */
    public function import(bool $bulk = false): bool
    {

        $recordsCount = $this->getRecordsCount();

        if ($recordsCount > DataImporter::MAX_SYNCHRONOUS_IMPORT_SIZE && ! $bulk) {
            throw new DataImportTooLargeException($recordsCount.' records is more than the maximum allowed of '.DataImporter::MAX_SYNCHRONOUS_IMPORT_SIZE);
        }

        $result = $this->importer->import();

        if (! $result) {
            throw new ImportFailedException('Import failed');
        }

        $validationErrors = $this->getValidationErrors();

        if ($this->getRecordsCount() == 0) {
            throw new DataImportNoRecordsImportedException('No records were imported', $validationErrors);
        }

        if ($validationErrors) {
            throw new DataImportValidationErrorsException(
                count($validationErrors) == $this->getRecordsCountByLooping() ? 'All records failed to import' : 'Some records failed to import',
                $validationErrors
            );
        }

        return true;
    }

    public function sendsImportEmail(): bool
    {
        return $this->importer->sendsImportEmail();
    }

    private function getValidationErrors(): array
    {
        return $this->importer->getValidationErrors();
    }

    private function getRecordsCountByLooping(): int
    {
        return $this->importer->recordsCountByLooping;
    }
}
