<?php

namespace App\Managers;

use App\Data\BlemishedProductData;
use App\Data\CreateInventoryAdjustmentData;
use App\Data\CreateInventoryAdjustmentFromBlemishedProductData;
use App\Exceptions\BlemishedSkuAlreadyExistsException;
use App\Models\InventoryAdjustment;
use App\Models\Product;
use App\Models\ProductBlemished;
use App\Services\Product\ProductDuplicator;
use Illuminate\Support\Facades\DB;
use Spatie\LaravelData\DataCollection;
use Throwable;

class BlemishedProductManager
{
    protected ProductDuplicator $duplicator;

    protected InventoryAdjustmentManager $InventoryAdjustmentManager;

    public function __construct(
        ProductDuplicator $duplicator,
        InventoryAdjustmentManager $InventoryAdjustmentManager
    ) {
        $this->duplicator = $duplicator;
        $this->InventoryAdjustmentManager = $InventoryAdjustmentManager;
    }

    /**
     * @throws Throwable
     */
    public function createBlemishedProduct(
        BlemishedProductData $data,
        bool $addStockToExisting = false
    ): Product {
        return DB::transaction(function () use ($data, $addStockToExisting) {
            $originalProduct = Product::findOrFail($data->derived_from_product_id);

            if (!($blemishedProduct = Product::whereSku($data->blemished_sku)->first())) {
                $blemishedProduct = $this->duplicator->duplicate($originalProduct, [
                    'sku' => $data->blemished_sku,
                    'type' => Product::TYPE_BLEMISHED,
                    'mpn' => $originalProduct->sku,
                ],
                    [ProductDuplicator::DUPLICATE_SUPPLIERS]
                );

                $this->createProductBlemishedPivot($data, $blemishedProduct);
            } else if (!$addStockToExisting) {
                throw new BlemishedSkuAlreadyExistsException('Blemished product with SKU ' . $blemishedProduct->sku . ' already exists.');
            }

            $this->handleInventoryAdjustments($data, $originalProduct, $blemishedProduct);

            return $blemishedProduct;
        });
    }

    /**
     * @throws Throwable
     */
    private function handleInventoryAdjustments(
        BlemishedProductData $data,
        Product $originalProduct,
        Product $blemishedProduct
    ): void {
        if ($data->adjust_inventory_for_original_product) {
            $this->adjustInventory(
                $data->adjustment_data,
                $originalProduct->id,
                InventoryAdjustment::TYPE_DECREASE
            );
        }

        if ($data->adjust_inventory) {
            $this->adjustInventory(
                $data->adjustment_data,
                $blemishedProduct->id,
                $data->adjustment_type
            );
        }
    }

    /**
     * @throws Throwable
     */
    private function adjustInventory(
        CreateInventoryAdjustmentFromBlemishedProductData $adjustmentData,
        int $productId,
        string $adjustmentType
    ): void {
        $data = CreateInventoryAdjustmentData::from([
            'product_id' => $productId,
            'adjustment_date' => $adjustmentData->adjustment_date,
            'warehouse_id' => $adjustmentData->warehouse_id,
            'quantity' => $adjustmentData->quantity,
            'unit_cost' => $adjustmentData->unit_cost,
            'notes' => $adjustmentData->notes,
            'adjustment_type' => $adjustmentType,
            'link_type' => $adjustmentData->link_type,
            'link_id' => $adjustmentData->link_id,
        ]);
        if ($adjustmentType === InventoryAdjustment::TYPE_INCREASE) {
            $data->unit_cost = $adjustmentData->unit_cost;
        }
        $this->InventoryAdjustmentManager->createAdjustment($data);
    }

    private function createProductBlemishedPivot(
        BlemishedProductData $data,
        Product $blemishedProduct
    ): void {
        ProductBlemished::create([
            'product_id' => $blemishedProduct->id,
            'derived_from_product_id' => $data->derived_from_product_id,
            'condition' => $data->condition,
            'reference' => $data->reference,
        ]);
    }
}
