<?php

namespace App\Mail;

use App\Models\User;
use Illuminate\Auth\Authenticatable;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Mail\Mailable;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Str;

/**
 * Class DataTableExportMail
 */
class DataTableExportMail extends Mailable implements ShouldQueue
{
    use Queueable, SerializesModels;

    private ?string $filePath;

    private string $resource;

    /**
     * @var User|Authenticatable|mixed
     */
    private User $user;

    public bool $failed = false;

    /**
     * DataTableExportMail constructor.
     */
    public function __construct(User $user, $resource, ?string $filePath = null)
    {
        $this->user = $user;
        $this->resource = $resource;
        $this->filePath = $filePath;
    }

    public function build(): self
    {
        $count = str_word_count($this->resource);

        if ($count === 1) {
            $resource = Str::plural($this->resource);
        } else {
            $resource = $this->resource;
        }

        $fileNameParts = explode('/', $this->filePath);
        $fileName = $fileNameParts[count($fileNameParts) - 1];

        return $this
            ->from(config('MAIL_FROM_ADDRESS', 'no-reply@sku.io'))
            ->markdown($this->failed ? 'mail.exports.failed' : 'mail.exports.index', [
                'resource' => $resource,
                'user' => $this->user,
                'filename' => $fileName,
                'failed' => $this->failed,
                'resourceHeadline' => strtolower(Str::headline($resource)),
            ])
            ->subject(
                $this->failed ? 'File Export Failed.' : 'Your file from '.config('app.name')
            );
    }

    public function failed(): self
    {
        $this->failed = true;

        return $this;
    }
}
