<?php

namespace App\Lib\SphinxSearch;

use App\Lib\Inspect\Inspect;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;

class SphinxSearch
{
    public static $connection = null;

    public static $join_count = 0;

    public static function connection()
    {
        if (is_null(static::$connection)) {
            static::$connection = DB::connection(config('sphinx.connection'));
        }

        return static::$connection;
    }

    public static function enabled()
    {
        // look in the configuration if Sphinx is enabled
        return config('sphinx.enabled');
    }

    public static function indexEnabled($index_name)
    {
        // look in the configuration if a given
        // Sphinx index is enabled
        return static::enabled() && $index_name !== false && in_array($index_name, static::indexes());
    }

    public static function indexes()
    {
        // return an array of indexes enabled according
        // to the configuration
        // Example:
        // SPHINX_INDEXES=sku_products,sku_product_attributes
        $prefix = config('sphinx.index_prefix');

        return array_map(function ($index) use ($prefix) {
            return $prefix.$index;
        }, array_map('trim', explode(',', config('sphinx.indexes'))));
    }

    public static function getRawSelect($index_name, array $args = [])
    {
        $retrieve_columns = Arr::get($args, 'columns', ['id']);
        $gateway_db = Arr::get($args, 'gateway_db', config('sphinx.gateway_db'));
        $gateway_table = Config::get(
            'sphinx.index_config.'.$index_name.'.gateway_table',
            $index_name
        );

        $select = implode(', ', $retrieve_columns).' FROM '.$gateway_db.'.'.$gateway_table;

        return $select;
    }

    public static function getRawQueryString(array $args)
    {
        $query = Arr::get($args, 'query', '');
        $query = str_replace(';', ' ', $query);
        $mode = trim(strtolower(Arr::get($args, 'mode', 'all')));
        $offset = Arr::get($args, 'offset', '0');
        $limit = Arr::get($args, 'limit', '1000');

        if ($mode != 'extended') {
            //header("X-Query: $query");
            $query = static::removeExtendedSyntax($query);
            //header("X-Query-sanitized: $query");
        }

        // header('X-Query-complete: '.
        //     $query.';mode='.$mode.';offset='.
        //     $offset.';limit='.$limit);

        return $query.';mode='.$mode.';offset='.
            $offset.';limit='.$limit;
    }

    public static function filter($query_builder, $index_name, array $args)
    {
        // Takes a query builder and adds a condition filtering by ID of records
        // found in the search.
        $query = Arr::get($args, 'query', null);

        $query = str_replace(';', ' ', $query);

        if (empty($query) || empty($index_name) ||
         ! static::indexEnabled($index_name)) {
            return $query_builder;
        }

        $fk_field = Arr::get($args, 'fk_field', 'id'); // what to match on MariaDB's side

        // Create new \Illuminate\Database\Query\Builder with current wheres in a group
        $new_query = $query_builder->getQuery()->forNestedWhere();
        $new_query->addNestedWhereQuery($query_builder->getQuery());

        //\App\Lib\Inspect\Inspect::dump($query_builder->getQuery()->orders);
        $new_query->orders = $query_builder->getQuery()->orders;

        // Add the new query in another group
        $new_query->whereIn(
            $fk_field,
            function ($q) use ($index_name, $args) {
                $q->select(\DB::raw(self::getRawSelect($index_name)))
                    ->where('query', self::getRawQueryString($args));
            }
        );

        // Replace \Illuminate\Database\Eloquent\Builder's internal
        // \Illuminate\Database\Query\Builder with the new one
        $query_builder->setQuery($new_query);

        return $query_builder;
    }

    public static function join($index_name, $query_builder, array $args)
    {
        // Takes a query builder and performs an inner join against a Sphinx
        // search.
        $query = Arr::get($args, 'query', null);

        $query = str_replace(';', ' ', $query);

        if (empty($query) || empty($index_name) ||
         ! static::indexEnabled($index_name)) {
            return $query_builder;
        }

        static::$join_count++;
        $gateway_db = Arr::get($args, 'gateway_db', config('sphinx.gateway_db'));

        $mode = Arr::get($args, 'mode', 'all');

        $offset = Arr::get($args, 'offset', '0');
        // offset should be usually 0, actual offset for pagination should be
        // managed by the query builder

        $limit = Arr::get($args, 'limit', '1000');
        // Max search results.  Actual limit for the query should
        // be managed by the query builder as always.

        $fk_field = Arr::get($args, 'fk_field', 'id'); // what to match on MariaDB's side

        $field_weights = Arr::get($args, 'field_weights', null);
        $fieldweights = $field_weights ? ';fieldweights='.$field_weights : '';

        $index_gateway_alias = 'sphinx_'.static::$join_count;

        $gateway_table = Config::get(
            'sphinx.index_config.'.$index_name.'.gateway_table',
            $index_name
        );

        $query_builder->join(
            $gateway_db.'.'.$gateway_table.' AS '.$index_gateway_alias,
            function ($join) use ($index_gateway_alias, $fk_field, $query, $mode, $offset, $limit, $fieldweights) {
                $join->on($index_gateway_alias.'.id', '=', $fk_field)
                    ->where(
                        $index_gateway_alias.'.query',
                        $query.';mode='.$mode.';offset='.
                               $offset.';limit='.$limit.$fieldweights
                    );
            }
        );

        return $query_builder;
    }

    public static function installEngine($connection)
    {
        // Installs SphinxSE engine in a given MariaDB connection.
        try {
            $result = $connection->statement(DB::raw("INSTALL SONAME 'ha_sphinx';"));

            return true;
        } catch (\Exception $e) {
            return false;
        }
    }

    public static function setupGateway($connection)
    {
        // Installs SphinxSE engine in a given MariaDB connection.
        try {
            $result = $connection->statement(DB::raw('CREATE DATABASE '.config('sphinx.gateway_db').';'));

            return true;
        } catch (\Exception $e) {
            // Do not count "database exist" (Error code 1007) as an error.
            return strpos($e->getMessage(), '1007') !== false;
        }
    }

    public static function setupGatewayTable($index_name)
    {
        // Creates a MariaDB table that will be used as a proxy by SphinxSE
        // storage engine inside MariaDB queries.
        $setup_sql = Config::get('sphinx.index_config.'.$index_name.'.setup_sql', false);

        if (empty($setup_sql)) {
            return false;
        }
        if (is_string($setup_sql)) {
            $setup_sql = [$setup_sql];
        }

        try {
            $connection = DB::connection('mysql');
            foreach ($setup_sql as $sql) {
                // $connection->getPdo()->query(DB::raw($sql))->fetchAll();
                $connection->getPdo()->query(DB::raw($sql)->getValue(DB::getQueryGrammar()))->fetchAll();
            }
        } catch (\Exception $e) {
            echo $e->getMessage()."\n";

            return false;
        }

        return true;
    }

    public static function rebuild($index_name)
    {
        $rebuild_sql = Config::get('sphinx.index_config.'.$index_name.'.rebuild_sql', false);
        if (empty($rebuild_sql)) {
            return false;
        }

        try {
            $connection = DB::connection('mysql');
            //Inspect::dump($connection);

            $cursor = $connection->cursor($rebuild_sql);

            $get_chunk = function ($cursor, $qty) {
                $count = 0;
                $chunk = [];
                while ($count < $qty && $cursor->valid()) {
                    $chunk[] = $cursor->current();
                    $cursor->next();
                    $count++;
                }

                return empty($chunk) ? false : $chunk;
            };

            // Inspect::dump($cursor);
            // Inspect::dump($get_chunk($cursor, 3));

            // $source_records = $connection->select(DB::raw($rebuild_sql));

            // $chunks = array_chunk($source_records, 100);

            while ($records = $get_chunk($cursor, 100)) {
                if (empty($columns)) {
                    $columns = implode(', ', array_keys((array) Arr::get($records, '0', [])));
                }

                if (empty($records)) {
                    exit('No records');
                }
                if (empty($columns)) {
                    continue;
                }

                // SphinxSearch is not friendly with prepared statements,
                // specially when performing updates of several records
                // at once.
                $records = array_map(function ($el) {
                    $r = array_map(function ($field) {
                        return
                            is_null($field) ? "'0'" :
                                //"'".str_replace("'", "\\'", $field)."'";
                                str_replace(['-', '=', '/'], '_',
                                    static::connection()->getPdo()->quote($field)
                                );
                    }, array_values((array) $el));

                    return implode(',', (array) $r);
                }, $records);
                $sphinxql = "REPLACE INTO $index_name ($columns) VALUES (";
                $sphinxql .= implode('),(', $records);
                $sphinxql .= ')';
                //dump($sphinxql);
                static::execute($sphinxql);
                //echo '.';
            }

            // echo("\n");
            // $connection->statement(DB::raw( "COMMIT;" ));
            return true;
        } catch (\Exception $e) {
            echo $e->getMessage()."\n";

            return false;
        }
    }

    public static function execute($query, $params = [])
    {
        $params = array_map(function ($e) {
            return is_null($e) ? "'0'" : $e;
        }, $params);

        return static::connection()->statement($query, $params);
    }

    public static function update($index_name, $record_provider_callback, $timestamp_field = 'updated_at')
    {
        // use $record_provider_callback to retrieve recently updated records
        // and use them to update the index specified in $index_name
    }

    public static function updateRecord($index_name, $record_data)
    {
        // save updated info for that record into index
        if (empty($record_data['id']) || empty($index_name) ||
         ! static::indexEnabled($index_name)) {
            return false;
        }
        $columns = implode(', ', array_keys($record_data));
        $values = array_values($record_data);
        $sphinxql = "REPLACE INTO $index_name ($columns) VALUES (?".str_repeat(', ?', count($values) - 1).')';
        static::execute($sphinxql, $values);

        return true;
    }

    public static function deleteRecord($index_name, $record_id)
    {
        // delete record
        if (empty($record_id) || empty($index_name) ||
         ! static::indexEnabled($index_name)) {
            return false;
        }
        $sphinxql = "DELETE FROM $index_name WHERE id = ".intval($record_id);
        static::execute($sphinxql);

        return true;
    }

    public static function removeExtendedSyntax($query)
    {
        // SphinxSE is tricky because the query itself goes with this format:
        //     WHERE QUERY="query_string;mode=all;limit=1000;offset=0"
        // The query string itself must only parse advanced modifiers if the mode
        // is "extended", but parsing the query occurs before finding out the
        // mode, so a query like this: "some words and a hyphen at the end -"
        // will fail because in extended mode the hyphen works as a modifier and
        // expects a word after it.
        // Being the mode "all" we would expect the parser to just allow the
        // hyphen, but it doesn't, and the query fails with a SphinSearch parse
        // error.
        // The only solution is to transform the query into something that will
        // pass the parsing phase, like removing the trailing hyphens and other
        // keywords.  Keywords are described here:
        //
        //     http://sphinxsearch.com/docs/current.html#extended-syntax

        $query = trim(strtolower($query)); // gets rid of MAYBE, NEAR, SENTENCE, PARAGRAPH
        /* | - ! * @ ~ / < = ^ $ */
        $query = rtrim($query, '/');
        //$query = str_replace("|", "_", $query);
        $query = str_replace('=', '_', $query);
        $query = str_replace('/', '_', $query);
        $query = str_replace('-', '_', $query);
        $query = trim($query, '-');

        return $query.' ?';
    }
}
