<?php
/*
Copyright (c) 2008 Sebastián Grignoli
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:
1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.
3. Neither the name of copyright holders nor the names of its
   contributors may be used to endorse or promote products derived
   from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL COPYRIGHT HOLDERS OR CONTRIBUTORS
BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.
*/

/**
 * @author   "Sebastián Grignoli" <grignoli@gmail.com>
 *
 * @version  2.0
 *
 * @link     https://github.com/neitanod/inspect
 *
 * @example  https://github.com/neitanod/inspect
 *
 * @license  Revised BSD
 */

namespace App\Lib\Inspect;

class Inspect
{
    public static function view($label, $val = '__undefin_e_d__')
    {
        if ($val == '__undefin_e_d__') {
            /* The first argument is not the label but the
               variable to inspect itself, so we need a label.
               Let's try to find out it's name by peeking at
               the source code.
            */
            $val = $label;

            $bt = debug_backtrace();
            $src = file($bt[0]['file']);
            $line = $src[$bt[0]['line'] - 1];

            // let's match the function call and the last closing bracket
            preg_match("#Inspect::view\((.+)\)#", $line, $match);

            /* let's count brackets to see how many of them actually belongs
               to the var name
               Eg:   die(inspect($this->getUser()->hasCredential("delete")));
                      We want:   $this->getUser()->hasCredential("delete")
            */
            $max = strlen($match[1]);
            $varname = '';
            $c = 0;
            for ($i = 0; $i < $max; $i++) {
                if ($match[1][$i] == '(') {
                    $c++;
                } elseif ($match[1][$i] == ')') {
                    $c--;
                }
                if ($c < 0) {
                    break;
                }
                $varname .= $match[1][$i];
            }
            $label = $varname;
        }

        // now the actual function call to the inspector method,
        // passing the var name as the label:
        dInspect::dump($label, $val, 10);

        return $val;
    }

    public static function dump($label, $val = '__undefin_e_d__')
    {
        if ($val == '__undefin_e_d__') {
            /* The first argument is not the label but the
               variable to inspect itself, so we need a label.
               Let's try to find out it's name by peeking at
               the source code.
            */
            $val = $label;

            $bt = debug_backtrace();
            $src = file($bt[0]['file']);
            $line = $src[$bt[0]['line'] - 1];

            // let's match the function call and the last closing bracket
            preg_match("#Inspect::dump\((.+)\)#", $line, $match);

            /* let's count brackets to see how many of them actually belongs
               to the var name
               Eg:   die(inspect($this->getUser()->hasCredential("delete")));
                      We want:   $this->getUser()->hasCredential("delete")
            */
            $max = strlen($match[1]);
            $varname = '';
            $c = 0;
            for ($i = 0; $i < $max; $i++) {
                if ($match[1][$i] == '(') {
                    $c++;
                } elseif ($match[1][$i] == ')') {
                    $c--;
                }
                if ($c < 0) {
                    break;
                }
                $varname .= $match[1][$i];
            }
            $label = $varname;
        }

        // now the actual function call to the inspector method,
        // passing the var name as the label:
        cInspect::dump($label, $val, 10);

        return $val;
    }

    public static function get($label, $val = '__undefin_e_d__')
    {
        if ($val == '__undefin_e_d__') {
            /* The first argument is not the label but the
               variable to inspect itself, so we need a label.
               Let's try to find out it's name by peeking at
               the source code.
            */
            $val = $label;

            $bt = debug_backtrace();
            $src = file($bt[0]['file']);
            $line = $src[$bt[0]['line'] - 1];

            // let's match the function call and the last closing bracket
            preg_match("#Inspect::get\((.+)\)#", $line, $match);

            /* let's count brackets to see how many of them actually belongs
               to the var name
               Eg:   die(inspect($this->getUser()->hasCredential("delete")));
                      We want:   $this->getUser()->hasCredential("delete")
            */
            $max = strlen($match[1]);
            $varname = '';
            $c = 0;
            for ($i = 0; $i < $max; $i++) {
                if ($match[1][$i] == '(') {
                    $c++;
                } elseif ($match[1][$i] == ')') {
                    $c--;
                }
                if ($c < 0) {
                    break;
                }
                $varname .= $match[1][$i];
            }
            $label = $varname;
        }

        // now the actual function call to the inspector method,
        // passing the var name as the label:
        $output = cInspect::get($label, $val, 10);

        return $output;
    }

    public static function getHtml($label, $val = '__undefin_e_d__')
    {
        if ($val == '__undefin_e_d__') {
            /* The first argument is not the label but the
               variable to inspect itself, so we need a label.
               Let's try to find out it's name by peeking at
               the source code.
            */
            $val = $label;

            $bt = debug_backtrace();
            $src = file($bt[0]['file']);
            $line = $src[$bt[0]['line'] - 1];

            // let's match the function call and the last closing bracket
            preg_match("#Inspect::getHtml\((.+)\)#", $line, $match);

            /* let's count brackets to see how many of them actually belongs
               to the var name
               Eg:   die(inspect($this->getUser()->hasCredential("delete")));
                      We want:   $this->getUser()->hasCredential("delete")
            */
            $max = strlen($match[1]);
            $varname = '';
            $c = 0;
            for ($i = 0; $i < $max; $i++) {
                if ($match[1][$i] == '(') {
                    $c++;
                } elseif ($match[1][$i] == ')') {
                    $c--;
                }
                if ($c < 0) {
                    break;
                }
                $varname .= $match[1][$i];
            }
            $label = $varname;
        }

        // now the actual function call to the inspector method,
        // passing the var name as the label:
        return dInspect::get($label, $val, 10);
    }
}

class dInspect
{
    public static function dump($label, &$val, $max_recursion)
    {
        $ipath = (substr($label, 0, 1) == '$') ? $label : '$'.$label;
        $bt = debug_backtrace();
        $o = '<div class="dinspect">'.self::ddump($val, ['ipath' => $ipath,
            'label' => $label,
            'max_recursion' => $max_recursion,
        ]).
             '<div class="di-footer">Called from: <span class="di-label">'.$bt[1]['file'].
             '</span>, line '.$bt[1]['line'].
             '<span class="di-credits">Inspector v1.0 by <a href="mailto:grignoli@at.gmail.com">Sebasti&aacute;n Grignoli</a> (c) 2010 </span></div></div>'."\n";
        echo $o;
    }

    public static function get($label, &$val, $max_recursion)
    {
        $ipath = (substr($label, 0, 1) == '$') ? $label : '$'.$label;
        $bt = debug_backtrace();
        $o = self::$script;
        $o .= self::$styles;
        $o .= '<div class="dinspect">'.self::ddump($val, ['ipath' => $ipath,
            'label' => $label,
            'max_recursion' => $max_recursion,
        ]).
              '<div class="di-footer">Called from: <span class="di-label">'.$bt[1]['file'].
              '</span>, line '.$bt[1]['line'].
              '<span class="di-credits">Inspector v1.0 by <a href="mailto:grignoli@at.gmail.com">Sebasti&aacute;n Grignoli</a> (c) 2010 </span></div></div>'."\n";

        return $o;
    }

    private static $script = '
  <script>
  function toggle(elm)
  {
    var elm = document.getElementById(elm);
    if(elm.style.visibility!="visible"){
        elm.style.visibility="visible";
        elm.style.display="block";
    }else{
        elm.style.visibility="hidden";
        elm.style.display="none";
    }
  }
  </script>
  ';

    private static $styles = '
  <style>
  .dinspect {
          color: black;
          background-color:#F0F0F0;
          border:  1px solid;
          padding: 2px 0 0 3px;
          margin: 2px 3px 2px 0px;
          font: 12px/13px Arial, Tahoma, Verdana;
      text-align: left;
  }
  .dinspect .property-name,
  .dinspect .method-syntax {
        color: royalblue;
  }
  .dinspect .public-property-label,
  .dinspect .public-method-label {
        color: green;
  }
  .dinspect .private-property-label,
  .dinspect .private-method-label {
    color: red;
  }
  .dinspect .protected-property-label,
  .dinspect .protected-method-label {
    color: orange;
  }
  .dinspect span.method-definition {
          font-size: 10px;
          color: lightgray;
          //display: none;
  }
  .dinspect .di-label:hover span.method-definition {
          display: inline;
  }
  .di-body {
          color: black;
          background-color:#FAFAFA;
          margin-left: 3px;
          padding: 2px 0 0 3px;
          border-left: 1px dotted;
  }
  .di-node {
          visibility:hidden;
          display:none;
          /* background-color:white; */
          margin-left: 3px;
          padding-left: 3px;
          padding:2px 0 0px 10px;
  }
  .di-longtext {
          visibility:hidden;
          display:none;
          background:none repeat scroll 0 0 lightyellow;
          border:1px solid #808000;
          font:13px Ubuntu Mono, Courier New;
          margin:5px 5px 0 0;
          overflow:auto;
          padding:5px;
  }
  .di-label {
          color:navy;
          font:bold 13px/12px Ubuntu Mono, Courier New;
  }
  .di-value {
          font-weight: bold;
  }
  .di-clickable {
          cursor: pointer;
  }
  div.di-clickable:hover, li.di-clickable:hover {
          background-color: #BFDFFF;
  }
  span.di-ipath:hover {
          background-color: yellow;
  }
  .di-footer {
          border-top: 1px dashed;
          margin-top: 8px;
          padding: 3px;
  }
  .di-credits {
          float: right;
          margin-right: 3px;
  }
  </style>
  ';

    public static function echoScript()
    {
        static $script_shown = false;
        if ($script_shown) {
            return;
        }
        $script_shown = true;
        echo self::$script;
    }

    public static function echoStyles()
    {
        static $styles_shown = false;
        if ($styles_shown) {
            return;
        }
        $styles_shown = true;
        echo self::$styles;
    }

    private static function ipathlink($ipath)
    {
        return '<span style="cursor:pointer" class="di-ipath" onclick="return prompt(\'Current branch:\', \''.str_replace("'", "\&#39;", str_replace('"', '&quot;', str_replace('\\', '\\\\', str_replace('>', '&gt;', $ipath)))).'\') && false;">&gt;</span>';
    }

    public static function ddump(&$val, $params)
    {
        $ipath = empty($params['ipath']) ? null : $params['ipath'];
        $label = ! isset($params['label']) ? null : $params['label'];
        $type = empty($params['type']) ? null : $params['type'];
        $len = ! isset($params['len']) ? null : $params['len'];
        $units = empty($params['units']) ? null : $params['units'];
        $max_recursion = $params['max_recursion'] === null ? 2 : $params['max_recursion'];

        self::echoStyles();
        self::echoScript();
        if (is_null($type)) {
            $type = ucfirst(gettype($val));
        }
        $o = '';
        if ($max_recursion == 0) {
            $o .= self::drawNode('**MAXIMUM RECURSION LEVEL REACHED**', $label, $ipath, 'Array', count($val), 'elements');
        } elseif (is_array($val)) {
            $rndtag = md5(microtime().rand(0, 1000));
            $o .= '<div class="di-body di-clickable" onclick="toggle(\''.$rndtag.'\');"> '.self::ipathlink($ipath).' <span class="di-label">'.(! ($label === false) ? $label : '...').'</span> ('.$type.', '.count($val).
                 ' elements) <span class="di-value">...</span></div>'."\n".'<div id="'.$rndtag.
                 '" class="di-node">';
            foreach ($val as $k => $v) {
                $o .= self::ddump($val[$k], [
                    'ipath' => $ipath.(is_int($k) ? '['.$k.']' : '[\''.$k.'\']'),
                    'label' => $k,
                    'max_recursion' => $max_recursion - 1,
                ]);
            }
            $o .= "</div>\n";
        } elseif (is_object($val)) {
            $rndtag = md5(microtime().rand(0, 1000));
            $o .= '<div class="di-body di-clickable" onclick="toggle(\''.$rndtag.'\');"> '.self::ipathlink($ipath).' <span class="di-label">'.(! ($label === false) ? $label.' ' : '... ').
                 '</span> (Object <span class="di-value">'.get_class($val).
                 '</span>) <span class="di-value">...</span></div>'."\n".'<div id="'.$rndtag.
                 '" class="di-node">';

            //$ref = new ReflectionClass(get_class($val));
            $ref = new \ReflectionObject($val);

            // datos de la clase
            if ($ref->getFileName()) {
                $o .= self::drawNode($ref->getFileName().($ref->getStartLine() ? ', line '.
                                                                                 $ref->getStartLine() : ''), 'Definition:', $ref->getFileName(), '');
            }

            $parent = get_class($val);
            $ancestors = $parent;
            while ($parent = get_parent_class($parent)) {
                $ancestors = $parent.' > '.$ancestors;
            }
            if ($ancestors != get_class($val)) {
                $o .= self::drawNode($ancestors, 'Ancestors:', '>', '');
            }

            // para cada propiedad
            $props = $ref->getProperties();
            foreach ($props as $prop) {
                /*
                         $prop->setAccessible(true);

                        if($prop->isProtected()) $prop->setAccessible(true);
                */
                if ($prop->isPublic()) {
                    $prop_value = $prop->getValue($val);
                    $prop_type = ucfirst(gettype($prop_value));
                } else {
                    $prop_value = '';
                    $prop_type = ' ';
                }
                $o .= self::ddump($prop_value, [
                    'ipath' => $ipath.'->'.$prop->name,
                    'label' => 'Property:'.($prop->isPublic() ? ' <span class="public-property-label">public</span>' :
                        '').($prop->isPrivate() ? ' <span class="private-property-label">private</span>' : '').($prop->isProtected() ?
                        ' <span class="protected-property-label">protected</span>' : '').($prop->isStatic() ? ' <span class="static-property-label">static</span>' : '').' <span class="property-name">$'.$prop->name.'</span>',
                    'type' => $prop_type,
                    '',
                    'max_recursion' => $max_recursion - 1,
                ]);
            }

            // para cada método
            $methods = $ref->getMethods();
            foreach ($methods as $method) {
                $params = $method->getParameters();
                $strparams = [];
                $optional_params = false;
                foreach ($params as $param) {
                    $pdefault = null;
                    try {
                        $pdefault = $param->getDefaultValue();
                    } catch (\ReflectionException $e) {
                    }
                    $pname = $param->name;
                    if ($param->isOptional() && ! $optional_params) {
                        $optional_params = true;
                        $pname = '['.$pname;
                    }

                    $strparams[] = $pname.(isset($pdefault) ? ' = '.(is_array($pdefault) ? '[]' : (is_bool($pdefault) ? ($pdefault ? 'TRUE' : 'FALSE') : $pdefault)) : '');
                }
                $strparams = implode(', ', $strparams);
                $emptystring = '';
                $methodSyntax = $method->name.'('.$strparams.
                        ($optional_params ? ']' : '').')';

                $o .= self::drawNode($emptystring, 'Method: '.($method->isPublic() ? ' <span class="public-method-label">public</span>' :
                                                '').($method->isPrivate() ? ' <span class="private-method-label">private</span>' : '').($method->isProtected() ?
                                                ' <span class="protected-method-label">protected</span>' : '').($method->isStatic() ? ' <span class="static-method-label">static</span>' : '').' <span class="method-syntax">'.$methodSyntax.'</span> <span class="method-definition">'.$method->getFileName().($method->getStartLine() ? ', line '.$method->getStartLine() : '').'</span>', $ipath.'->'.$methodSyntax, '');
                //print_r($method);
            }
            unset($ref);
            $o .= "</div>\n";
        //dInspect::objectStack("add",$val);
        } elseif (is_bool($val)) {
            $o .= '<div class="di-body"> '.self::ipathlink($ipath).' <span class="di-label">'.(! ($label === false) ?
          $label.' ' : '... ').'</span>('.$type.') <span class="di-value">'.($val ?
          'TRUE' : 'FALSE')."</span></div>\n";
        } elseif (is_string($val) && $type == 'String') {
            $len = strlen($val);
            $encoding = mb_detect_encoding($val, 'UTF-8, ISO-8859-1', true);
            if ($len < 60) {
                $type = $type.', '.$encoding;
                $o .= self::drawNode(htmlspecialchars($val, ENT_QUOTES, $encoding), $label, $ipath, $type, $len, is_null($units) ?
          'characters' : $units);
            } else {
                $rndtag = md5(microtime().rand(0, 1000));
                $o .= '<div class="di-body di-clickable" onclick="toggle(\''.$rndtag.'\');"> '.self::ipathlink($ipath).' <span class="di-label">'.(! ($label === false) ? $label : '...').'</span> (String, '.$encoding.', '.$len.' '.(is_null($units) ? 'characters' : $units).') <span class="di-value">'.htmlspecialchars(substr(
                    $val,
                    0,
                    60
                ), ENT_QUOTES, $encoding).'...</span></div>'."\n".'<pre id="'.$rndtag.'" class="di-longtext">'.htmlspecialchars($val, ENT_QUOTES, $encoding).
                   '</pre>';
            }
        } elseif (is_string($val)) {  /* && $type != 'String', we already know that. */
            $encoding = mb_detect_encoding($val, 'UTF-8, ISO-8859-1', true);
            $o .= self::drawNode(htmlspecialchars($val, ENT_QUOTES, $encoding), $label, $ipath, $type, $len, $units);
        } else {
            $o .= self::drawNode($val, $label, $ipath);
        }

        return $o;
    }

    private static function drawNode($val, $label = null, $ipath = '#', $type = null, $len = null, $units = null)
    {
        if (gettype($val) == 'boolean') {
            $val = $val ? 'TRUE' : 'FALSE';
        }
        $type = is_null($type) ? ucfirst(gettype($val)) : $type;
        $label = is_null($label) ? '...' : $label;
        $units = is_null($units) ? 'bytes' : $units;
        $len = is_null($len) ? '' : $len.' '.$units;
        $desc = '';
        if (trim($type)) {
            $desc = trim($len) ? $type.', '.$len : $type;
        }
        $desc = '('.$desc.')';
        if ($desc == '()') {
            $desc = '';
        }

        return '<div class="di-body">'.self::ipathlink($ipath).' <span class="di-label">'.$label.
           '</span> '.$desc.' <span class="di-value">'.$val."</span></div>\n";
    }
}

class cInspect
{
    protected static $indent = 0;

    public static function get($label, &$val, $max_recursion)
    {
        $ipath = (substr($label, 0, 1) == '$') ? $label : '$'.$label;
        $bt = debug_backtrace();
        $o = '// ********************************************** '."\n".
             self::ddump($val, ['ipath' => $ipath, 'label' => $label, 'max_recursion' => $max_recursion]).
             '// ******* ^^^ dump called from: '.$bt[1]['file'].', line '.$bt[1]['line']."\n";

        return $o;
    }

    public static function dump($label, &$val, $max_recursion)
    {
        $ipath = (substr($label, 0, 1) == '$') ? $label : '$'.$label;
        $bt = debug_backtrace();
        $o = '// ********************************************** '."\n".
             self::ddump($val, ['ipath' => $ipath, 'label' => $label, 'max_recursion' => $max_recursion]).
             '// ******* ^^^ dump called from: '.$bt[1]['file'].', line '.$bt[1]['line']."\n";
        echo $o;
    }

    public static function ddump(&$val, $params)
    {
        self::$indent++;
        $ipath = empty($params['ipath']) ? null : $params['ipath'];
        $label = ! isset($params['label']) ? null : $params['label'];
        $type = empty($params['type']) ? null : $params['type'];
        $len = ! isset($params['len']) ? null : $params['len'];
        $units = empty($params['units']) ? null : $params['units'];
        $max_recursion = $params['max_recursion'] === null ? 2 : $params['max_recursion'];

        if (is_null($type)) {
            $type = ucfirst(gettype($val));
        }
        $o = '';
        if ($max_recursion == 0) {
            $o .= self::drawNode('**MAXIMUM RECURSION LEVEL REACHED**', $label, $ipath, 'Array', count($val), 'elements');
        } elseif (is_array($val)) {
            $o .= self::indent().(
                ! ($label === false) ? $label : ': '
            ).' ('.$type.', '.count($val).
            " elements) : \n";
            foreach ($val as $k => $v) {
                $o .= self::ddump($val[$k], [
                    'ipath' => $ipath.(is_int($k) ? '['.$k.']' : '[\''.$k.'\']'),
                    'label' => '['.$k.']',
                    'max_recursion' => $max_recursion - 1,
                ]);
            }
        } elseif (is_object($val)) {
            $o .= self::indent().(! ($label === false) ? $label.' ' : ': ').
            '(Object '.get_class($val).
            ") :\n";
            self::$indent++;
            //$ref = new ReflectionClass(get_class($val));
            $ref = new \ReflectionObject($val);

            // datos de la clase
            if ($ref->getFileName()) {
                $o .= self::drawNode($ref->getFileName().($ref->getStartLine() ? ', line '.
                                                                                 $ref->getStartLine() : ''), 'Definition:', $ref->getFileName(), '');
            }

            $parent = get_class($val);
            $ancestors = $parent;
            while ($parent = get_parent_class($parent)) {
                $ancestors = $parent.' > '.$ancestors;
            }
            if ($ancestors != get_class($val)) {
                $o .= self::drawNode($ancestors, 'Ancestors:', '>', '');
            }

            // para cada propiedad
            $props = $ref->getProperties();
            foreach ($props as $prop) {
                /*
                         $prop->setAccessible(true);

                        if($prop->isProtected()) $prop->setAccessible(true);
                */
                if ($prop->isPublic()) {
                    $prop_value = $prop->getValue($val);
                    $prop_type = ucfirst(gettype($prop_value));
                } else {
                    $prop_value = '';
                    $prop_type = ' ';
                }
                self::$indent--;
                $o .= self::ddump($prop_value, [
                    'ipath' => $ipath.'->'.$prop->name,
                    'label' => 'Property:'.($prop->isPublic() ? ' public' :
                        '').($prop->isPrivate() ? ' private' : '').($prop->isProtected() ?
                        ' protected' : '').($prop->isStatic() ? ' static' : '').' $'.$prop->name,
                    'type' => $prop_type,
                    '',
                    'max_recursion' => $max_recursion - 1,
                ]);
                self::$indent++;
            }

            // para cada método
            $methods = $ref->getMethods();
            foreach ($methods as $method) {
                $params = $method->getParameters();
                $strparams = [];
                $optional_params = false;
                foreach ($params as $param) {
                    $pdefault = null;
                    try {
                        $pdefault = $param->getDefaultValue();
                    } catch (\ReflectionException $e) {
                    }
                    $pname = $param->name;
                    if ($param->isOptional() && ! $optional_params) {
                        $optional_params = true;
                        $pname = '['.$pname;
                    }
                    $strparams[] = $pname.(isset($pdefault) ? ' = '.(is_array($pdefault) ? 'Array' : $pdefault) : '');
                }
                $strparams = implode(', ', $strparams);
                $emptystring = '';
                $methodSyntax = $method->name.'('.$strparams.
                        ($optional_params ? ']' : '').')';

                $o .= self::drawNode($emptystring, 'Method:'.($method->isPublic() ? ' public' :
                                                '').($method->isPrivate() ? ' private' : '').($method->isProtected() ?
                                                ' protected' : '').($method->isStatic() ? ' static' : '').' '.$methodSyntax, $ipath.'->'.$methodSyntax, '');
                //print_r($method);
            }
            unset($ref);
            //cInspect::objectStack("add",$val);
            self::$indent--;
        } elseif (is_bool($val)) {
            $o .= self::indent().(! ($label === false) ?
          $label.' ' : '... ').'('.$type.') '.($val ?
          'TRUE' : 'FALSE')."\n";
        } elseif (is_string($val) && $type == 'String') {
            $len = strlen($val);
            $encoding = mb_detect_encoding($val, 'UTF-8, ISO-8859-1', true);
            $type .= ', '.$encoding;
            $o .= self::drawNode('"'.$val.'"', $label, $ipath, $type, $len, is_null($units) ?
        'characters' : $units);
        } elseif (is_string($val)) {  /* && $type != 'String', we already know that. */
            $o .= self::drawNode('"'.$val.'"', $label, $ipath, $type, $len, $units);
        } else {
            $o .= self::drawNode($val, $label, $ipath);
        }
        self::$indent--;

        return $o;
    }

    private static function drawNode($val, $label = null, $ipath = '#', $type = null, $len = null, $units = null)
    {
        if (gettype($val) == 'boolean') {
            $val = $val ? 'TRUE' : 'FALSE';
        }
        $type = is_null($type) ? ucfirst(gettype($val)) : $type;
        $label = is_null($label) ? '...' : $label;
        $units = is_null($units) ? 'bytes' : $units;
        $len = is_null($len) ? '' : $len.' '.$units;
        $desc = '';
        if (trim($type)) {
            $desc = trim($len) ? $type.', '.$len : $type;
        }
        $desc = '('.$desc.')';
        if ($desc == '()') {
            $desc = '';
        }

        return self::indent().$label.' '.$desc.($val == '' ? '' : ' => '.$val)."\n";
    }

    private static function indent()
    {
        return str_repeat('    ', self::$indent);
    }
}
