<?php

namespace App\Lib\DataList;

use App\Lib\Xlsx2Csv\Xlsx2Csv;

class DataListFactory
{
    public static function load(string $fileName): DataListInterface
    {
        switch (self::guessFileType($fileName)) {
            case 'xlsx':
            case 'xls':
                $tempFile = self::getTempFileName('csv');
                Xlsx2Csv::convert($fileName, $tempFile);

                return new LeagueCsvDataList($tempFile);
            case 'csv':
                return new LeagueCsvDataList($fileName);
            case 'tsv':
                // TODO-SAM
                return new LeagueCsvDataList($fileName, "\t");
            default:
                throw new DataListException('Unsupported file type');
        }
    }

    protected static function convertEncoding(
        string $inputFileName,
        string $inputEncoding,
        string $outputFileName,
        string $outputEncoding,
    ) {
        // TODO there is an ineffiecenty in this library, perhaps file_get_contents() is
        //      been called more often than needed

        $outputFileContents = mb_convert_encoding(
            file_get_contents($inputFileName),
            $outputEncoding,
            $inputEncoding,
        );
        file_put_contents($outputFileName, $outputFileContents);
    }

    protected static function guessEncoding(string $fileName): string
    {
        $contents = file_get_contents($fileName);
        $encoding = mb_detect_encoding(
            $contents,
            [
                'UTF-8',
                'UTF-16',
            ]
        );

        return $encoding;
    }

    protected static function guessFileType(string $fileName): string
    {
        $pathinfo = pathinfo($fileName);
        if (! isset($pathinfo['extension'])) {
            throw new DataListException('Cannot guessFileType');
        }

        switch (strtolower($pathinfo['extension'])) {
            case 'xlsx': // Excel (OfficeOpenXML) Spreadsheet
            case 'xlsm': // Excel (OfficeOpenXML) Macro Spreadsheet (macros will be discarded)
            case 'xltx': // Excel (OfficeOpenXML) Template
            case 'xltm': // Excel (OfficeOpenXML) Macro Template (macros will be discarded)
                return 'xlsx';
            case 'xls': // Excel (BIFF) Spreadsheet
            case 'xlt': // Excel (BIFF) Template
                return 'xls';
            case 'csv':
                return 'csv';
            case 'tsv':
            case 'txt':
                return 'tsv';
            default:
                throw new DataListException('Cannot guessFileType');
        }
    }

    protected static function getTempFileName(?string $extension = null): string
    {
        $tempFileName = tempnam(sys_get_temp_dir(), 'DataListFactory');

        if ($extension) {
            $tempFileName .= '.'.$extension;
        }

        return $tempFileName;
    }
}
