<?php

namespace App\Lib\DBSchema;

use DB;

class DBSchema
{
    protected $db;

    public function __construct($db)
    {
        $this->db = $db;

        return $this;
    }

    public function getDatabases()
    {
        $databases = [];
        $databases_result = $this->db->select(DB::raw('SHOW DATABASES;'));
        foreach ($databases_result as $table) {
            foreach ($table as $key => $value) {
                $databases[] = $value;
            }
        }

        return $databases;
    }

    public function getTables($databaseName)
    {
        // get original active DB
        $current_db = $this->db->getDatabaseName();

        // temporarily change to target DB:
        $this->db->setDatabaseName($databaseName);

        $tables = [];
        $tables_result = $this->db->select(DB::raw('SHOW TABLES'));
        foreach ($tables_result as $table) {
            foreach ($table as $key => $value) {
                $tables[] = $value;
            }
        }

        // restore original active DB
        $this->db->setDatabaseName($current_db);

        return $tables;
    }

    public function generateForDatabases($databaseNames)
    {
        $schema = ['databases' => []];
        foreach ($databaseNames as $alias => $db) {
            $schema['databases'][$alias] = $this->generateForDatabase($db);
        }

        return $schema;
    }

    public function generateForDatabase($databaseName)
    {
        $tables = $this->getTables($databaseName);

        $schema = ['tables' => []];
        foreach ($tables as $table) {
            $schema['tables'][$table] = $this->generateForTable($databaseName, $table);
        }

        return $schema;
    }

    public function generateForTable($databaseName, $table)
    {
        $db_data = (array) $this->db->selectOne('SELECT * FROM information_schema.SCHEMATA WHERE schema_name = ?', [$databaseName]);
        $db_charset = $db_data['DEFAULT_CHARACTER_SET_NAME'];
        $db_collation = $db_data['DEFAULT_COLLATION_NAME'];

        $table_data = (array) $this->db->selectOne('SHOW TABLE STATUS WHERE Name = ?', [$table]);

        if (! $table_data) {
            return false;  // Table not found
        }

        $table_collation = $table_data['Collation'];
        $table_engine = $table_data['Engine'];

        $col_rows = $this->db->select('SELECT * FROM information_schema.COLUMNS WHERE TABLE_SCHEMA = ? AND TABLE_NAME = ?', [
            $databaseName,
            $table,
        ]);

        $fields = [];
        foreach ($col_rows as $col_data_object) {
            $col_data = (array) $col_data_object;
            $field = [];
            $field['type'] = $col_data['COLUMN_TYPE'];
            $col_data['IS_NULLABLE'] != 'NO' ?: $field['not_null'] = true;

            if (! empty($col_data['COLLATION_NAME']) && $col_data['COLLATION_NAME'] !== $table_collation) {
                $field['collation'] = $col_data['COLLATION_NAME'];
            }
            if (isset($col_data['COLUMN_DEFAULT'])) {
                $field['default'] = $col_data['COLUMN_DEFAULT'];
            }
            if (! empty($col_data['EXTRA'])) {
                $field['extra'] = $col_data['EXTRA'];
            }

            if (! empty($col_data['COLUMN_COMMENT'])) {
                $extraData = json_decode($col_data['COLUMN_COMMENT'], true);
                // inspect($extraData);
                if (! empty($extraData)) {
                    if (array_key_exists('comment', $extraData)) {
                        $field['comment'] = $extraData['comment'];
                    }
                    if (array_key_exists('type', $extraData)) {
                        $field['type'] = $extraData['type'];
                    }
                    if (array_key_exists('private', $extraData)) {
                        $field['private'] = $extraData['private'];
                    }
                } else {
                    $field['comment'] = $col_data['COLUMN_COMMENT'];
                }
            }

            // When only 'type' defined convert field data to string
            // if (count($field)==1 && isset($field['type'])) {
            //     $field = $field['type'];
            // }

            $fields[$col_data['COLUMN_NAME']] = $field;
        }
        $schema['columns'] = $fields;

        $key_rows = $this->db->select('SHOW KEYS FROM '.$table);

        $primary = [];
        $unique = [];
        $keys = [];

        foreach ($key_rows as $key_row) {
            $v = (array) $key_row;
            if ($v['Key_name'] == 'PRIMARY') {
                $primary[$v['Seq_in_index'] - 1] = $v['Column_name'];
                $schema['columns'][$v['Column_name']]['primary'] = true;
            } elseif (empty($v['Non_unique'])) {
                $unique[$v['Key_name']][$v['Seq_in_index'] - 1] = $v['Column_name'];
                $schema['columns'][$v['Column_name']]['index'] = true;
            } else {
                $keys[$v['Key_name']][$v['Seq_in_index'] - 1] = $v['Column_name'];
                $schema['columns'][$v['Column_name']]['unique'] = true;
            }
        }

        ksort($primary);
        if (count($primary) == 1) {
            $primary = $primary[0];
        }
        foreach ($unique as $k => $v) {
            ksort($unique[$k]);
            if (count($unique[$k]) == 1) {
                $unique[$k] = $unique[$k][0];
            }
        }
        foreach ($keys as $k => $v) {
            ksort($keys[$k]);
            if (count($keys[$k]) == 1) {
                $keys[$k] = $keys[$k][0];
            }
        }

        if (is_string($primary) || count($primary)) {
            $schema['primary'] = $primary;
        }
        if (count($unique)) {
            $schema['unique'] = $unique;
        }
        if (count($keys)) {
            $schema['index'] = $keys;
        }

        return $schema;
    }
}
