<?php
/*
Copyright (c) 2021 Sebastián Grignoli
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:
1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.
3. Neither the name of copyright holders nor the names of its
   contributors may be used to endorse or promote products derived
   from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL COPYRIGHT HOLDERS OR CONTRIBUTORS
BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.
*/

/**
 * @author   "Sebastián Grignoli" <grignoli@gmail.com>
 *
 * @version  1.0
 *
 * @link     https://github.com/neitanod/arrayinsql
 *
 * @license  Revised BSD
 */

/*
This class is a rudimentary attempt to solve the memory exhaustion problem
caused by the abuse of arrays.
If you have complex logic that is very hard to refactor because of some
mishandling of massive arrays, like traversing them once and again, performing
several transformations to thir data, etc, and replacing the arrays with
generators is too hard, try replacing them with instances of this class.
They will use mysql temporary tables instead of plain memory.  They'll be
slow, but won't fail from memory exhaustion.

Simply replace:

        $mydata = [];

with:

        $pdo = new \PDO("mysql:dbname=test;host=127.0.0.1", "root", "mypass");
        // (In Laravel:   $pdo = \DB::connection()->getPdo();)
        $mydata = new \Neitanod\ArrayInSql\ArrayInSql($pdo);

You'll not be able to do sort(), ksort(), et al, serialize() and unserialize(),
but everything else should work just the same.
 */

namespace App\Lib\ArrayInSql;

use ArrayAccess;
use Countable;
use Iterator;

class ArrayInSql implements ArrayAccess, Countable, Iterator
{ //, SeekableIterator,  Serializable {
    // const SORT_BY_FIFO = 1;
    // const SORT_BY_KEY = 2;
    // const SORT_BY_VALUE = 3;
    // const SORT_BY_VALUE_NATURAL = 4;
    // const SORT_BY_VALUE_NATURAL_CASE_SENSITIVE = 5;

    // protected $sortBy = self::SORT_BY_FIFO;
    protected $PDO = null;

    protected $tmpTableName = null;

    protected $currentKey = null;

    protected $currentValue = null;

    protected $maxKey = -1;

    public function __construct($pdo)
    {
        $this->PDO = $pdo;

        // Decide on a temp table name and create it
        $this->tmpTableName = 'array_in_sql_'.substr(md5(date('YmdHis').rand()), 0, 8);
        //$sql = " CREATE TABLE ".$this->tmpTableName."
        $sql = ' CREATE TEMPORARY TABLE '.$this->tmpTableName.'
                (
                    id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
                    array_key LONGBLOB NOT NULL,
                    array_value LONGBLOB NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `array_key_index` (array_key(10))
                    -- ,KEY `array_value_index` (array_value(10))
                )';
        $this->query($sql);
    }

    protected function query($sql, $data = [])
    {
        // echo($sql);
        $stmt = $this->PDO->prepare($sql);
        $stmt->execute($data);
        if ($this->PDO->errorCode() != '00000') {
            $err = $this->PDO->errorInfo();
            throw new \Exception('ArrayInSql error: ['.$err[0].'] '.$err[1].' '.$err[2].' '.$sql);
        }

        return $stmt;
    }

    public function append($value): void
    {
        $sql = '
            INSERT INTO '.$this->tmpTableName.' (array_key, array_value) VALUES (?,?);';
        $this->query($sql, [++$this->maxKey, serialize($value)]);
    }

    // public function asort(): void {
    //     $this->sortBy = self::SORT_BY_VALUE;
    // }

    public function count(): int
    {
        $sql = 'SELECT count(*) FROM '.$this->tmpTableName.';';
        $s = $this->query($sql);

        return intval($s->fetchColumn(0));
    }

    public function current()
    {
        [$this->currentKey, $this->currentValue] = $this->generator->current();

        return $this->currentValue;
    }

    public function getArrayCopy()
    {
        $sql = 'SELECT array_key, array_value FROM '.$this->tmpTableName.';';
        $s = $this->query($sql);
        $all = $s->fetchAll();
        $generator = function () use ($all) {
            foreach ($all as $record) {
                yield $record['array_key'] => unserialize($record['array_value']);
            }
        };

        return $generator;
    }

    // public function getFlags(): void {}

    public function key()
    {
        [$this->currentKey, $this->currentValue] = $this->generator->current();

        return $this->currentKey;
    }

    // public function ksort(): void {
    //     $this->sortBy = self::SORT_BY_KEY;
    // }

    // public function natcasesort(): void {
    //     $this->sortBy = self::SORT_BY_VALUE_NATURAL;
    // }

    // public function natsort(): void {
    //     $this->sortBy = self::SORT_BY_VALUE_NATURAL_CASE_SENSITIVE;
    // }

    public function next(): void
    {
        $this->generator->next();
    }

    public function offsetExists($offset): bool
    {
        $sql = 'SELECT COUNT(*) FROM '.$this->tmpTableName.' WHERE (array_key = ?);';
        $s = $this->query($sql, [$offset]);

        return (bool) intval($s->fetchColumn(0));
    }

    public function offsetGet($offset)
    {
        $sql = 'SELECT array_value FROM '.$this->tmpTableName.' WHERE (array_key = ?);';
        $s = $this->query($sql, [$offset]);

        return unserialize($s->fetchColumn(0));
    }

    public function offsetSet($offset, $newval): void
    {
        if (is_null($offset)) {
            $this->append($newval);
        }
        // TODO: implement
        if ((is_int($offset) && $offset >= $this->maxKey)) {
            $this->maxKey = $offset;
        }
        $sql = 'REPLACE INTO '.$this->tmpTableName.' (array_key, array_value) VALUES (?,?);';
        $this->query($sql, [$offset, serialize($newval)]);
    }

    public function offsetUnset($offset): void
    {
        $sql = 'DELETE * FROM '.$this->tmpTableName.' WHERE array_key = ?;';
        $this->query($sql, [$offset]);
    }

    public function rewind(): void
    {
        $sql = 'SELECT array_key, array_value FROM '.$this->tmpTableName.';';
        $s = $this->query($sql);
        $this->generator = (function () use ($s) {
            while ($record = $s->fetch()) {
                yield [$record['array_key'], unserialize($record['array_value'])];
            }
        })();
    }

    // public function seek($position): void {
    //     // throw new \Exception("seek not implemented");
    // }

    // public function serialize(): string {
    //     // throw new \Exception("serialize not implemented");
    // }

    // public function setFlags(string $flags): void {}

    // public function uasort(string $cmp_function): void {
    //     // throw new \Exception("uasort not implemented");
    // }

    // public function uksort(string $cmp_function): void {
    //     // throw new \Exception("uasort not implemented");
    // }

    // public function unserialize($serialized): string {
    //     // throw new \Exception("unserialize not implemented");
    // }

    public function valid(): bool
    {
        return $this->generator->valid();
    }
}
