<?php

namespace App\Jobs\WooCommerce;

use App\Events\SalesChannelListingLinked;
use App\Helpers;
use App\Integrations\Channel;
use App\Models\Attribute;
use App\Models\Product;
use App\Models\ProductListing;
use App\Models\SalesChannel;
use App\Models\SalesChannelType;
use App\Models\Setting;
use Exception;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Str;

class GetProducts implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * @var SalesChannel
     */
    protected $salesChannel;

    /**
     * Create a new job instance.
     */
    public function __construct(SalesChannel $salesChannel)
    {
        $this->salesChannel = $salesChannel;
    }

    /**
     * Execute the job.
     *
     * @throws Exception
     */
    public function handle(): void
    {
        /**
         * Get instance of channel type.
         */
        $woo_channel = Channel::getInstance(SalesChannelType::TYPE_WOOCOMMERCE, $this->salesChannel->credentials);

        /**
         * Get current Currency of WooCommerce store.
         */
        $woo_currency = $woo_channel->getCurrentCurrency();

        /**
         * Fetch products.
         */
        $products_generator = $woo_channel->getProducts(['per_page' => 10]);

        $defaultWeightUnit = Helpers::setting(Setting::KEY_DEFAULT_WEIGHT_UNIT, null, true);
        $defaultDimensionUnit = Helpers::setting(Setting::KEY_DEFAULT_DIMENSION_UNIT, null, true);

        foreach ($products_generator as $products) {
            /**
             * Basic attributes.
             */
            $description_attribute = Attribute::firstOrCreate(
                ['name' => 'description'],
                [
                    'type' => Attribute::TYPE_LONGTEXT,
                    'has_options' => false,
                ]
            );

            $image_attribute = Attribute::firstOrCreate(
                ['name' => 'image'],
                [
                    'type' => Attribute::TYPE_LONGTEXT,
                    'has_options' => false,
                ]
            );

            $condition_attribute = Attribute::firstOrCreate(
                ['name' => 'condition'],
                [
                    'type' => Attribute::TYPE_STRING,
                    'has_options' => false,
                ]
            );

            $currency_attribute = Attribute::firstOrCreate(
                ['name' => 'currency'],
                [
                    'type' => Attribute::TYPE_STRING,
                    'has_options' => false,
                ]
            );

            foreach ($products as $woo_product) {
                /**
                 * Get from DB or New object of ProductListing by "sales channel" and "product listing id".
                 */
                $product_listing = ProductListing::firstOrNew([
                    'sales_channel_id' => $this->salesChannel->id,
                    'sales_channel_listing_id' => $woo_product['id'],
                ]);
                /**
                 * Check if product listing exists on DB.
                 * Get Product by id (product listing exists), by sku (product listing not exists) or New object.
                 */
                if ($product_listing->exists) {
                    $product = Product::findOrNew($product_listing->product_id);
                } else {
                    $product = Product::whereNotNull('sku')->where('sku', $woo_product['sku'] ?? null)->firstOrNew([]);

                    if ($product->exists) {
                        event(new SalesChannelListingLinked($product, $this->salesChannel));
                    }
                }

                /**
                 * Fill Product attributes and save on DB.
                 */
                $product->type = $woo_product['virtual'] ? Product::TYPE_VIRTUAL : Product::TYPE_STANDARD;
                $product->sku = $woo_product['sku'];
                $product->name = $woo_product['title'];
                $product->weight = $woo_product['weight'] ?? 0;
                $product->weight_unit = $defaultWeightUnit;
                $product->length = $woo_product['dimensions']['length'] ?: 0;
                $product->width = $woo_product['dimensions']['width'] ?: 0;
                $product->height = $woo_product['dimensions']['height'] ?: 0;
                $product->dimension_unit = $woo_product['dimensions']['unit'] ?: $defaultDimensionUnit;

                $product->case_dimension_unit = $defaultDimensionUnit;
                $product->case_weight_unit = $defaultWeightUnit;
                $product->save();

                /**
                 * Fill Product Listing attributes and save on DB.
                 */
                $product_listing->product_id = $product->id;
                $product_listing->listing_sku = $woo_product['sku'];
                $product_listing->price = floatval($woo_product['price']);
                $product_listing->quantity = $woo_product['stock_quantity'];
                $product_listing->save();

                /**
                 * Sync Product attributes and values.
                 */
                $product_image = $woo_product['featured_src'] ?? null;
                if ($product_image) {
                    $product_image = Str::startsWith($woo_product['featured_src'], ['http:', 'https:']) ?
            $woo_product['featured_src'] :
            'http:'.$woo_product['featured_src'];
                }

                $product->productAttributes()->syncWithoutDetaching(
                    [
                        $description_attribute->id => ['value_longtext' => $woo_product['description']],
                        $image_attribute->id => ['value_longtext' => $product_image],
                        $condition_attribute->id => ['value_string' => null],
                        $currency_attribute->id => ['value_string' => $woo_currency],
                    ]
                );

                /**
                 * Store variant attributes and values.
                 */
                $variant_attributes = [];
                foreach ($woo_product['attributes'] as $attribute) {
                    if ($attribute['variation']) {
                        $variant_attribute = Attribute::firstOrCreate(
                            ['name' => $attribute['name']],
                            [
                                'type' => Attribute::TYPE_STRING,
                                'has_options' => true,
                            ]
                        );

                        $variant_attributes[$attribute['slug']] = $variant_attribute;

                        foreach ($attribute['options'] as $value) {
                            $variant_attribute->values()->firstOrCreate(['value' => $value]);
                        }
                    }
                }

                /**
                 * Store Product variations by same way of parent product.
                 */
                foreach ($woo_product['variations'] ?? [] as $woo_variation) {
                    $variation_product_listing = ProductListing::firstOrNew([
                        'sales_channel_id' => $this->salesChannel->id,
                        'sales_channel_listing_id' => $woo_variation['id'],
                    ]);

                    /**
                     * Check if product listing exists on DB.
                     * Get Product by id (product listing exists) or by sku (product listing not exists) or New object.
                     */
                    if ($variation_product_listing->exists) {
                        $product_variation = Product::findOrNew($variation_product_listing->product_id);
                    } else {
                        $product_variation = Product::whereNotNull('sku')->where('sku', $woo_variation['sku'] ?: null)->firstOrNew([]);
                    }

                    /**
                     * Fill Product attributes and save on DB.
                     */
                    $product_variation->parent_id = $product->id;
                    $product_variation->brand_id = $product_brand->id ?? null;
                    $product_variation->type = $woo_variation['virtual'] ? Product::TYPE_VIRTUAL : Product::TYPE_STANDARD;
                    $product_variation->sku = $woo_variation['sku'] ?: null;
                    $product_variation->weight = $woo_variation['weight'] ?: 0;
                    $product_variation->weight_unit = $defaultWeightUnit;
                    $product_variation->length = $woo_variation['dimensions']['length'] ?: 0;
                    $product_variation->width = $woo_variation['dimensions']['width'] ?: 0;
                    $product_variation->height = $woo_variation['dimensions']['height'] ?: 0;
                    $product_variation->dimension_unit = $woo_variation['dimensions']['unit'] ?: $defaultDimensionUnit;

                    $product_variation->case_dimension_unit = $defaultDimensionUnit;
                    $product_variation->case_weight_unit = $defaultWeightUnit;

                    $product_variation->save();

                    /**
                     * Fill Product Listing attributes and save on DB.
                     */
                    $variation_product_listing->product_id = $product_variation->id;
                    $variation_product_listing->listing_sku = $woo_variation['sku'] ?: null;
                    $variation_product_listing->price = floatval($woo_variation['price']);
                    $variation_product_listing->quantity = $woo_variation['stock_quantity'];
                    $variation_product_listing->save();

                    /**
                     * Store Variant Image.
                     */
                    $variant_image = $woo_variation['image'][0]['src'] ?? null;
                    if ($variant_image) {
                        $variant_image = Str::startsWith($woo_variation['image'][0]['src'], ['http:', 'https:']) ?
              $woo_variation['image'][0]['src'] :
              'http:'.$woo_variation['image'][0]['src'];

                        $product_variation->productAttributes()->syncWithoutDetaching([$image_attribute->id => ['value_longtext' => $variant_image]]);
                    }

                    /**
                     * Store product/variation attributes and values.
                     */
                    foreach ($woo_variation['attributes'] as $attribute) {
                        /**
                         * WooCommerce add "pa_" As "Parent Attribute" to variant attribute slug.
                         */
                        $attribute_slug = Str::replaceFirst('pa_', '', $attribute['slug']);

                        $product_variation->productAttributes()->syncWithoutDetaching([$variant_attributes[$attribute_slug]->id => ['value_string' => $attribute['option']]]);
                    }
                }

                \App\Models\WooCommerce\Product::updateOrCreate(['id' => $woo_product['id']], $woo_product);
            }
        }
    }
}
