<?php

namespace App\Jobs\Starshipit;

use App\Models\Starshipit\StarshipitOrder;
use App\Notifications\MonitoringMessage;
use Exception;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Notification;

class ProcessStarshipitWebhookJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public function __construct(private readonly array $request)
    {
        $this->onQueue('fulfillments');
    }

    /**
     * @throws Exception
     */
    public function handle(): void
    {
        $request = $this->request;
        if (!isset($request['order_number'])) {
            throw new Exception('Starshipit webhook request does not contain order_number.  Request: ' . json_encode($request));
        }
        $starshipitOrders = StarshipitOrder::with(['salesOrderFulfillment.salesOrder'])->where('order_number', $request['order_number'])->get();

        if ($starshipitOrders->count() > 1) {
            $message = "The Starshipit order {$request['order_number']} has {$starshipitOrders->count()} fulfillments\n".config('app.url');
            Notification::route('slack', config('slack.debugging'))->notify(new MonitoringMessage($message));
        }

        $starshipitOrders->each(function (StarshipitOrder $starshipitOrder) use ($request) {
            if (! $starshipitOrder->salesOrderFulfillment) {
                $message = "The Starshipit order {$starshipitOrder->order_number} does not belong to a fulfillment\n".config('app.url');
                Notification::route('slack', config('slack.debugging'))->notify(new MonitoringMessage($message));

                // TODO: we should delete it, but basically we want to know why this is happening
                //                $starshipitOrder->delete();
                return; // continue
            }

            if (! $starshipitOrder->salesOrderFulfillment->salesOrder) {
                $message = "The fulfillment {$starshipitOrder->salesOrderFulfillment->id} does not belong to a sales order\n".config('app.url');
                Notification::route('slack', config('slack.debugging'))->notify(new MonitoringMessage($message));

                // TODO: we should delete it, but basically we want to know why this is happening
                //                $starshipitOrder->salesOrderFulfillment->delete();
                return; // continue
            }

            $trackingInfo = [
                'status' => $request['tracking_status'],
                'carrier_name' => $request['carrier_name'],
                'carrier_service' => $request['carrier_service'],
                'tracking_number' => $request['tracking_number'],
                'shipment_date' => $request['shipment_date'],
                'tracking_status' => $request['tracking_status'],
                'last_updated_date' => $request['last_updated_date'],
                'received_by' => 'Webhook',
            ];

            // update sales order fulfillment
            GetTrackingJob::addTrackingToSalesOrderFulfillment($starshipitOrder->salesOrderFulfillment, $trackingInfo);
            // update Starshipit order
            GetTrackingJob::updateStarshipitOrder($starshipitOrder, $trackingInfo, saveFulfillmentCost: true);
        });
    }
}
