<?php

namespace App\Jobs\Starshipit;

use App\Helpers;
use App\Integrations\Starshipit;
use App\Models\IntegrationInstance;
use App\Models\SalesChannel;
use App\Models\SalesOrderFulfillment;
use App\Models\Setting;
use App\Models\ShippingMethodMappingsSkuToShippingProviderMethod;
use App\Models\Starshipit\StarshipitOrder;
use App\Services\SalesOrder\FulfillSalesOrderService;
use App\Services\Shopify\ShopifySubmitTrackingInfo;
use Carbon\Carbon;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Log;

class GetTrackingJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected IntegrationInstance $integrationInstance;

    protected array $salesOrderFulfillmentIds;

    protected bool $fetchStarshipitOrder;

    /**
     * Create a new job instance.
     *
     * @return void
     */
    public function __construct(IntegrationInstance $integrationInstance, array|int $salesOrderFulfillmentIds = [], bool $fetchStarshipitOrder = true)
    {
        $this->integrationInstance = $integrationInstance;
        $this->salesOrderFulfillmentIds = Arr::wrap($salesOrderFulfillmentIds);
        $this->fetchStarshipitOrder = $fetchStarshipitOrder;
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        set_time_limit(0);

        $starshipit = new Starshipit($this->integrationInstance);

        SalesOrderFulfillment::query()
            ->where('fulfillment_type', SalesOrderFulfillment::TYPE_STARSHIPIT)
            ->when(empty($this->salesOrderFulfillmentIds), function (Builder $builder) {
                $builder->where('status', SalesOrderFulfillment::STATUS_SUBMITTED);
                $builder->whereNotNull('packing_slip_printed_at');
            })
            ->when(! empty($this->salesOrderFulfillmentIds), function (Builder $builder) {
                $builder->whereIn('id', $this->salesOrderFulfillmentIds);
            })
            ->each(function (SalesOrderFulfillment $fulfillment) use ($starshipit) {
                $starshipitOrder = $fulfillment->getShippingProviderOrder();
                if (! $starshipitOrder) {
                    return;
                }

                do {
                    $response = $starshipit->getTracking($starshipitOrder->order_number);
                    if ($response->statusCode == 429) {
                        sleep(2);

                        continue;
                    }

                    try {
                        if (($response->body['success'] ?? false) && isset($response->body['results']['tracking_number'])) {
                            $result = $response->body['results'];
                            $trackingInfo = [
                                'status' => $result['order_status'],
                                'carrier_name' => $result['carrier_name'],
                                'carrier_service' => $result['carrier_service'],
                                'tracking_number' => $result['tracking_number'],
                                'shipment_date' => $result['shipment_date'],
                                'tracking_status' => $result['tracking_status'],
                                'last_updated_date' => $result['last_updated_date'],
                                'received_by' => 'GetTrackingJob',
                            ];

                            // update sales order fulfillment
                            static::addTrackingToSalesOrderFulfillment($fulfillment, $trackingInfo, $this->integrationInstance);
                            // update Starshipit order
                            static::updateStarshipitOrder($starshipitOrder, $trackingInfo, $this->fetchStarshipitOrder, $this->integrationInstance, true);
                        }
                    } catch (\Throwable $exception) {
                        dump($exception->getMessage());
                        Log::debug($exception->getMessage(), $exception->getTrace());
                    }
                    break;
                } while (true);
            });
    }

    public static function addTrackingToSalesOrderFulfillment(SalesOrderFulfillment $salesOrderFulfillment, array $tracking, ?IntegrationInstance $starshipitIntegrationInstance = null)
    {
        if ($tracking['status'] === 'Cancelled') {
            $salesOrderFulfillment->fulfilled_at = null;
            $salesOrderFulfillment->status = SalesOrderFulfillment::STATUS_CANCELED;
            $salesOrderFulfillment->save();

            return;
        }

        if (in_array($tracking['status'], ['Manifested', 'Delivered', 'Dispatched'])) {
            $starshipitIntegrationInstance = $starshipitIntegrationInstance ?: IntegrationInstance::with([])->starshipit()->firstOrFail();

            // get/create shipping method mapping
            $shippingProviderMethodMap = ShippingMethodMappingsSkuToShippingProviderMethod::query()->firstOrCreate([
                'shipping_provider_id' => $starshipitIntegrationInstance->id,
                'shipping_provider_carrier' => $tracking['carrier_name'],
                'shipping_provider_method' => $tracking['carrier_service'] ?? SalesChannel::UNSPECIFIED_SHIPPING_METHOD, ]);

            // update sales order fulfillment
            $salesOrderFulfillment->tracking_number = $tracking['tracking_number'] ?: $salesOrderFulfillment->tracking_number;
            $salesOrderFulfillment->fulfilled_shipping_method_id = $shippingProviderMethodMap->shipping_method_id;
            $salesOrderFulfillment->fulfilled_shipping_method = $tracking['carrier_name'];
            $salesOrderFulfillment->fulfilled_at = Carbon::parse($tracking['shipment_date'], Helpers::setting(Setting::KEY_DEFAULT_TIMEZONE));
            $salesOrderFulfillment->status = SalesOrderFulfillment::STATUS_FULFILLED;
            $salesOrderFulfillment->cost = static::getFulfillmentCostFromPriceBreakdown($tracking['price_breakdown'] ?? null);
            $salesOrderFulfillment->save();

            // submit tracking to the sales channel
            try {
                $response = ShopifySubmitTrackingInfo::factory($salesOrderFulfillment)?->submit();
                if (! $response['success']) {
                    $errorMessage = "Can't submit tracking to the sales channel: ".$salesOrderFulfillment->salesOrder->sales_order_number.', response: '.json_encode($response);
                    //Notification::route('slack', config('slack.debugging'))->notify(new MonitoringMessage($errorMessage));
                    Log::debug($errorMessage);
                }
            } catch (\Exception $e) {
                $errorMessage = "Can't submit tracking to the sales channel: ".$salesOrderFulfillment->salesOrder->sales_order_number.', exception: '.$e->getMessage();
                //Notification::route('slack', config('slack.debugging'))->notify(new MonitoringMessage($errorMessage));
                Log::debug($errorMessage, $e->getTrace());
            }

            // mark sales order as fulfilled
            FulfillSalesOrderService::make($salesOrderFulfillment->salesOrder)->updateFulfillmentStatus($salesOrderFulfillment->fulfilled_at);
        }
    }

    public static function updateStarshipitOrder(StarshipitOrder $starshipitOrder, array $tracking, bool $getStarshipitOrder = true, ?IntegrationInstance $starshipitIntegrationInstance = null, $saveFulfillmentCost = false)
    {
        $starshipitOrder->shipment = $tracking;
        $starshipitOrder->save();

        if ($getStarshipitOrder) {
            dispatch(new GetOrder($starshipitOrder, $starshipitIntegrationInstance, $saveFulfillmentCost));
        }
    }

    public static function getFulfillmentCostFromPriceBreakdown(?array $priceBreakdown): ?float
    {
        if (empty($priceBreakdown)) {
            return null;
        }

        $cost = $priceBreakdown['Total Charge'] ??
               $priceBreakdown['Calculated Price ex GST'] ??
               $priceBreakdown['Total Price'] ??
               $priceBreakdown['Total Shipping Cost'] ??
               null;

        return $cost ? floatval(preg_replace("/[^-\d.]/", '', $cost)) : null;
    }
}
