<?php

namespace App\Jobs\Shopify;

use App\Exceptions\IntegrationInstance\UncancellableShippingProviderOrderException;
use App\Models\IntegrationInstance;
use App\Models\Shopify\ShopifyOrder;
use App\Services\StockTake\OpenStockTakeException;
use Exception;
use Illuminate\Bus\Batchable;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldBeUnique;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use romanzipp\QueueMonitor\Traits\IsMonitored;
use Throwable;

/**
 * Class UpdateSalesOrderFromShopifyOrderJob.
 */
class UpdateSalesOrderFromShopifyOrderJob implements ShouldBeUnique, ShouldQueue
{
    use Batchable, Dispatchable, InteractsWithQueue, IsMonitored, Queueable, SerializesModels;

    public int $uniqueFor = 60 * 5;

    public function uniqueId(): string
    {
        return 'update-order-from-shopify-order-job:'.$this->integrationInstance->id.'-'.$this->order_name;
    }

    /**
     * ProcessOrders constructor.
     */
    public function __construct(
        protected IntegrationInstance|int $integrationInstance,
        protected string $order_name,
        protected array $options = []
    ) {
        $this->onQueue('salesOrderProcessing');
        $this->integrationInstance = $integrationInstance instanceof IntegrationInstance ? $integrationInstance : IntegrationInstance::query()->findOrFail($integrationInstance);
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function handle(): void
    {
        try {
            customlog('shopifyOrdersBenchmark', $this->order_name.': UpdateSkuOrder starting');
            $shopifyOrder = ShopifyOrder::getOrderFromOrderName($this->order_name, $this->integrationInstance);
            if ($shopifyOrder->needsUpdating()) {
                $shopifyOrder->updateSKUOrder();
            }
            else {
                customlog('shopifyOrdersBenchmark', $this->order_name.': UpdateSkuOrder not needed');
            }
        } catch (OpenStockTakeException $openStockTakeException) {
            /** @see #SKU-5240 for this the job should just return, not fail.
             *      The idea is that stock takes are expected so shouldn’t cause any failures.
             *      Stock take locks shouldn’t last too long
             */
        } catch (UncancellableShippingProviderOrderException){

        }
    }
}
