<?php

namespace App\Jobs\Shopify;

use App\Models\IntegrationInstance;
use App\Services\Shopify\ShopifyProductManager;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

class ShopifyGetProductVariantsJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $integrationInstance;

    protected $cursor;

    protected $productVariants;

    protected $currentlyAvailable; // Add this line

    /**
     * Create a new job instance.
     */
    public function __construct(
        IntegrationInstance $integrationInstance,
        ?string $cursor = null,
        array $productVariants = [],
        int $currentlyAvailable = 50 // Assign a default value if needed
    ) {
        $this->integrationInstance = $integrationInstance;
        $this->cursor = $cursor;
        $this->productVariants = $productVariants;
        $this->currentlyAvailable = $currentlyAvailable;
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        $shopify = new \App\Integrations\Shopify($this->integrationInstance);
        $productVariantGid = 'gid://shopify/ProductVariant/';

        $log = Log::channel('single')->info('Cursor: '.$this->cursor);

        while (true) {
            try {
                $batchSize = min($this->currentlyAvailable, 250);
                $data = $shopify->getProductVariants($batchSize, $this->cursor);
                $edges = $data['data']['productVariants']['edges'];
                $hasNextPage = $data['data']['productVariants']['pageInfo']['hasNextPage'];
                $this->cursor = $data['data']['productVariants']['pageInfo']['endCursor'];
                $queryCost = $data['extensions']['cost'];
                $productVariants = [];

                foreach ($edges as $edge) {
                    $productVariants[] = trim($edge['node']['id'], $productVariantGid);
                }

                $this->productVariants = array_merge($this->productVariants, $productVariants);

                $this->currentlyAvailable = $queryCost['throttleStatus']['currentlyAvailable'] ?? $this->currentlyAvailable;
                $delay = $shopify::getGraphqlTimeToRetry($queryCost);

                if ($delay > 0) {
                    $nextJob = new self(
                        $this->integrationInstance,
                        $this->cursor,
                        $this->productVariants,
                        $this->currentlyAvailable // Pass currentlyAvailable to the next job
                    );
                    dispatch($nextJob)->delay(now()->addSeconds($delay + 2));

                    return;
                }

                if (! $hasNextPage) {
                    break;
                }
            } catch (\Exception $e) {
                if (strpos($e->getMessage(), 'Throttled') !== false) {
                    $nextJob = new self(
                        $this->integrationInstance,
                        $this->cursor,
                        $this->productVariants,
                        $this->currentlyAvailable // Pass currentlyAvailable to the next job
                    );
                    dispatch($nextJob)->delay(now()->addSeconds(3));

                    return;
                } else {
                    throw $e;
                }
            }
        }

        if (! $hasNextPage) {
            echo ' count : '.count($this->productVariants);
            $productManager = new ShopifyProductManager($this->integrationInstance);
            $productManager->markProductVariantsActiveFromShopify($this->productVariants);
            $productManager->markProductVariantsRemovedThatExistInSkuButMissingFromShopify($this->productVariants);
        }
    }
}
