<?php

namespace App\Jobs\Shopify;

use App\Integrations\Shopify;
use App\Models\IntegrationInstance;
use App\Models\Shopify\ShopifyLocation;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

class ShopifyGetLocationsJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * @var IntegrationInstance
     */
    protected $integrationInstance;

    /**
     * Create a new job instance.
     *
     * @return void
     */
    public function __construct(IntegrationInstance $integrationInstance)
    {
        $this->integrationInstance = $integrationInstance;
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        $shopify = new Shopify($this->integrationInstance);
        $locationIds = [];
        foreach ($shopify->getLocations() as $location) {
            ShopifyLocation::query()->upsert(['integration_instance_id' => $this->integrationInstance->id, 'json_data' => json_encode($location)], []);
            $locationIds[] = $location['id'];
        }

        ShopifyLocation::query()
            ->where('integration_instance_id', $this->integrationInstance->id)
            ->whereNotIn('shopify_location_id', $locationIds)
            ->delete();

        // Add inventory locations to the integration instance
        $this->setInventoryLocationsToIntegrationInstance();

        // SUMMARY
        $this->summary['Locations Count'] = count($locationIds);
        // LOGGING
        $this->options = ['Integration Instance' => $this->integrationInstance->name.' ('.$this->integrationInstance->id.') '];
    }

    /**
     * Add inventory locations to the integration instance.
     *
     * @see SKU-3453
     */
    private function setInventoryLocationsToIntegrationInstance()
    {
        $integrationSettings = $this->integrationInstance->integration_settings;
        if(!isset($integrationSettings['inventory'])) {
            $integrationSettings['inventory'] = [];
        }

        $existedInventoryLocations = collect($integrationSettings['inventory']['locations'] ?? []);
        // to delete the removed/inactive shopify locations
        $inventoryLocations = collect([]);

        foreach (ShopifyLocation::with([])->where('integration_instance_id', $this->integrationInstance->id)->get() as $shopifyLocation) {
            if (! $shopifyLocation->active) {
                continue;
            }

            // If the location downloaded already exists, don’t change any master of stock/warehouse/etc. settings
            if ($existedInventoryLocation = $existedInventoryLocations->firstWhere('shopify_location_id', $shopifyLocation->shopify_id)) {
                $inventoryLocations->add($existedInventoryLocation);
            } else {
                /**
                 * If the location does not already exist, initialize with:
                 *  - master of stock: none
                 *  - warehouses: none.
                 */
                $inventoryLocations->add([
                    'id' => $shopifyLocation->shopify_location_id,
                    'name' => $shopifyLocation->name,
                    'masterOfStock' => IntegrationInstance::SETTINGS_OPTION_NAME_NEITHER,
                    'selectedWarehouses' => [],
                ]);
            }
        }
        // save integration locations
        $integrationSettings['inventory']['locations'] = $inventoryLocations->values()->toArray();
        $this->integrationInstance->integration_settings = $integrationSettings;
        $this->integrationInstance->save();
    }
}
