<?php

namespace App\Jobs\Shopify;

use App\Helpers;
use App\Importers\Parsers\FieldParserFactory;
use App\Jobs\CacheProductListingPriceJob;
use App\Jobs\GenerateCacheProductListingQuantityJob;
use App\Lib\SphinxSearch\SphinxSearch;
use App\Models\Attribute;
use App\Models\DataImportMapping;
use App\Models\IntegrationInstance;
use App\Models\ProductPricingTier;
use App\Models\SalesOrderLine;
use App\Models\Setting;
use App\Models\Shopify\ShopifyProduct;
use App\Models\Shopify\ShopifyWebhook;
use App\Models\Supplier;
use App\Models\SupplierPricingTier;
use App\Models\Warehouse;
use App\Repositories\WarehouseRepository;
use Illuminate\Bus\Batchable;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

/**
 * Class CreateSkuProducts.
 */
class ShopifyCreateSkuProducts implements ShouldQueue
{
    use Batchable, Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected IntegrationInstance $integrationInstance;

    protected array $options;

    private Collection $mappings;

    private string $temporaryProductsTable = 'temporary_shopify_products';

    private string $temporaryTagsTable = 'temporary_shopify_tags';

    private ?array $shopifyProductIds;

    /**
     * GetOrders constructor.
     */
    public function __construct(IntegrationInstance $integrationInstance, ?array $shopifyProductIds = null)
    {
        $this->integrationInstance = $integrationInstance;
        $this->shopifyProductIds = $shopifyProductIds;
    }

    public function handle(): void
    {
        set_time_limit(0);

        $this->mappings = collect(DataImportMapping::query()
            ->where('model', 'listings')
            ->where('integration_instance_id', $this->integrationInstance->id)
            ->value('mapping'));
        if ($this->mappings->isEmpty()) {
            throw new \Exception('the integration instance does not have mapping rules.');
        }
        // sku field
        if (! $skuMapping = $this->mappings->firstWhere('sku_field', 'sku')) {
            throw new \Exception('the mapping does not have sku field');
        }
        $skuListingField = $skuMapping['listing_field'];
        if (! empty($parsers = $skuMapping['parsers'])) {
            foreach ($parsers as $parser) {
                $skuListingField = FieldParserFactory::make($parser['rule'], $parser['args'], $this)->parseSqlQuery($skuListingField, 'shopify_products');
            }
        } else {
            $skuListingField = "`shopify_products`.`$skuListingField`";
        }

        // get the mapped supplier pricing tier
        $supplierPricingTier = null;
        $supplierPricingMapping = null;
        if ($supplierPricingMapping = $this->mappings->filter(fn ($m) => preg_match("/supplier_pricing\..*\.value/", $m['sku_field']))->first()) {
            $supplierPricingTierName = explode('.', $supplierPricingMapping['sku_field'])[1];
            $supplierPricingTier = SupplierPricingTier::query()->firstWhere('name', $supplierPricingTierName);
        } else {
            $supplierPricingTier = SupplierPricingTier::query()->firstWhere('is_default', true);
        }

        $count = 3000;
        do {
            $productsToCreate = ShopifyProduct::query()
                ->select('shopify_products.*', 'products.id as sku_product_id')
                ->where('integration_instance_id', $this->integrationInstance->id)
                ->whereNull('product')
                ->whereNotNull('shopify_products.unit_cost')
                ->where('removed_from_shopify', 0)
                ->whereRaw(DB::raw("$skuListingField != ''")->getValue(DB::getQueryGrammar()))
                            // existing skus just link them to product_listings and map sales order lines
                ->leftJoin('products', 'products.sku', '=', DB::raw($skuListingField));

            if ($this->shopifyProductIds) {
                $productsToCreate->whereIn('shopify_products.id', $this->shopifyProductIds);
            }

            $productsToCreate = $productsToCreate->forPage(1, $count)->get();

            if ($productsToCreate->isEmpty()) {
                break;
            }

            //          print $productsToCreate->count() . ' products to create/link' . "\n";
            //          print $productsToCreate->clone()->whereNull('products.id')->count() . ' products after existing skus' . "\n";

            $dataInsertProducts = [];
            $dataInsertTags = [];
            // parse products data to temporary tables data
            $productsToCreate->each(function (ShopifyProduct $product) use (&$dataInsertTags, &$dataInsertProducts) {
                //                print ($product->sku_product_id ? 'Linking ' : 'Creating ') . $product->sku . "\n";
                $insertProduct = [
                    'shopify_product_id' => $product->id,
                    'shopify_product_title' => $product->title,
                    'shopify_product_price' => $product->price,
                    'shopify_product_sku' => $product->sku,
                    'variant_id' => $product->variant_id,
                    'inventory_quantity' => $product->inventory_quantity,
                    'product_id' => $product->sku_product_id,
                    'product_exists' => is_null($product->sku_product_id) ? 0 : 1,
                ];

                foreach ($this->mappings as $mapping) {
                    $listingFieldValue = $product->parseRules($product->getValue($mapping['listing_field']), $mapping['parsers'] ?? []);
                    // add tags data
                    if ($mapping['sku_field'] == 'tags' && ! empty($listingFieldValue)) {
                        foreach (explode(', ', $listingFieldValue) as $tag) {
                            $dataInsertTags[] = ['shopify_product_id' => $product->id, 'tag' => $tag];
                        }
                    } else {
                        $insertProduct[$this->temporaryColumnName($mapping['listing_field'])] = $listingFieldValue;
                    }
                }

                $dataInsertProducts[] = $insertProduct;
            });

            DB::transaction(function () use ($dataInsertTags, $supplierPricingMapping, $supplierPricingTier, $dataInsertProducts) {
                /**
                 * if we notice a performance issue,
                 * we can create other temporary tables for images, pricing, attributes and others,
                 * by this way we can join and insert them together like tags.
                 * currently, the query adds one attribute(for example) to all products
                 */

                // create the temporary table and insert products data
                $this->insertProductsDataToTemporaryTable($dataInsertProducts);

                // create product brands and add the brand_id to the temporary table
                $this->insertBrandsDataToTemporaryTable();

                // insert products with basic fields
                $this->insertProducts();

                // add product images
                if ($this->mappings->whereIn('sku_field', ['image', 'image_url', 'other_images'])->isNotEmpty()) {
                    DB::insert($this->insertProductImagesQuery());
                }

                // add default supplier to products
                $this->addDefaultSupplierToProducts($supplierPricingTier, $supplierPricingMapping['listing_field'] ?? null);

                // add pricing to products
                $this->addPricingToProducts();

                // add attributes to products
                $this->addAttributesToProducts();

                // add tags to products
                $this->addTagsToProducts($dataInsertTags);

                // add main category to products
                $this->addMainCategoryToProducts();

                // insert product listings
                $this->insertProductListings();

                // map sales order lines
                $this->mapSalesOrderLines();
            });
        } while ($productsToCreate->count() == $count);

        SphinxSearch::rebuild('products');

        // cache quantity for product listings
        GenerateCacheProductListingQuantityJob::dispatchSync($this->integrationInstance);
        // cache price for product listings
        CacheProductListingPriceJob::dispatchSync($this->integrationInstance);
    }

    private function insertProductsDataToTemporaryTable(array $productsData)
    {
        // create the temporary table
        DB::statement($this->createTemporaryTableQuery());
        // DO NOT use truncate(or drop table), it causes a transaction exception
        DB::delete("DELETE FROM {$this->temporaryProductsTable}");
        // insert data to the temporary table
        $columns = $this->columnize(array_keys(reset($productsData)));
        $values = $this->parameterize($productsData);
        DB::insert("INSERT IGNORE INTO {$this->temporaryProductsTable} ($columns) VALUES $values");
    }

    private function createTemporaryTableQuery(): string
    {
        $customDatatype = ['body_html' => 'text', 'tags' => 'text'];
        $mappingFields = '';
        foreach ($this->mappings as $mapping) {
            $type = $customDatatype[$mapping['listing_field']] ?? 'varchar(255)';
            $column = $this->temporaryColumnName($mapping['listing_field']);
            $mappingFields .= "`$column` $type,";
        }
        $uniqueColumn = $this->temporaryColumnName($this->mappings->firstWhere('sku_field', 'sku')['listing_field']);

        $collate = config('database.connections.mysql.collation');

        return <<<SQL
            CREATE TEMPORARY TABLE IF NOT EXISTS {$this->temporaryProductsTable} (
                `shopify_product_id` bigint(20) unsigned, 
                `variant_id` bigint(20) unsigned, 
                `product_id` bigint(20) unsigned,
                `inventory_quantity` bigint(20),
                `shopify_product_title` varchar(255),
                `shopify_product_sku` varchar(255),
                `shopify_product_price` decimal(9,4),
                `product_exists` tinyint(1) default 0, 
                `supplier_id` bigint(20) unsigned,
                `supplier_product_id` bigint(20) unsigned, 
                `supplier_warehouse_id` bigint(20) unsigned, 
                `product_listing_id` bigint(20) unsigned,
                `brand_id` bigint(20) unsigned,
                `category_id` bigint(20) unsigned,
                $mappingFields
                PRIMARY KEY (`shopify_product_id`), INDEX(`variant_id`), UNIQUE (`$uniqueColumn`), INDEX (`product_id`), INDEX (`product_exists`), INDEX (`supplier_id`), INDEX(`supplier_id`, `product_id`), INDEX(`supplier_product_id`), INDEX(`supplier_warehouse_id`), INDEX(`product_listing_id`), INDEX (`category_id`)
            ) ENGINE=InnoDB DEFAULT COLLATE={$collate}
        SQL;
    }

    private function insertProductsQuery(): string
    {
        $productColumns = $this->mappings->whereIn('sku_field', (new \App\Models\Product())->getFillable());
        // add brand_id
        if ($this->mappings->firstWhere('sku_field', 'brand')) {
            $productColumns->push(['listing_field' => 'brand_id', 'sku_field' => 'brand_id']);
        }
        $listingColumns = $productColumns->pluck('listing_field')->map(fn ($c) => "`{$this->temporaryColumnName($c)}`");

        return <<<SQL
                INSERT INTO products (`type`, {$productColumns->pluck('sku_field')->implode(',')}, `created_at`, `updated_at`) 
                SELECT 'standard', {$listingColumns->implode(',')}, NOW(), NOW() FROM {$this->temporaryProductsTable} WHERE product_id IS NULL;
                SQL;
    }

    private function insertProductImagesQuery(): string
    {
        $primaryImageColumn = $this->mappings->whereIn('sku_field', ['image', 'image_url'])->first()['listing_field'] ?? null;
        if (! $primaryImageColumn) {
            $primaryImageColumn = $this->mappings->firstWhere('sku_field', 'other_images')->first()['listing_field'];
        }

        $unionOtherImagesSQL = '';
        foreach ($this->mappings->where('sku_field', 'other_images') as $otherImage) {
            // the image is not the primary image
            if ($otherImage['listing_field'] != $primaryImageColumn) {
                $unionOtherImagesSQL .= "UNION SELECT product_id, {$otherImage['listing_field']}, 0 FROM {$this->temporaryProductsTable} WHERE product_exists = 0\n";
            }
        }

        return <<<SQL
            INSERT IGNORE INTO product_images (product_id, url, is_primary, created_at, updated_at) 
            SELECT product_id, url, is_primary, NOW(), NOW() FROM (
                SELECT product_id, $primaryImageColumn url, 1 is_primary FROM {$this->temporaryProductsTable} WHERE product_exists = 0 
                {$unionOtherImagesSQL}
                ) images
            WHERE images.url IS NOT NULL
            SQL;
    }

    private function insertSupplierProductQuery(): string
    {
        $additionalColumns = [
            'default_supplier_sku' => 'supplier_sku',
            'default_supplier_leadtime' => 'leadtime',
            'default_supplier_moq' => 'minimum_order_quantity',
        ];
        $additionalMappedColumns = $this->mappings->whereIn('sku_field', array_keys($additionalColumns))->keyBy('sku_field');

        $additionalMapped = collect($additionalColumns)->only($additionalMappedColumns->keys());
        $supplierProductColumns = $additionalMapped->isNotEmpty() ? (','.$additionalMapped->values()->implode(',')) : '';
        $listingsColumns = $additionalMapped->isNotEmpty() ? (','.$additionalMapped->keys()->map(fn ($k) => "`{$this->temporaryColumnName($additionalMappedColumns[$k]['listing_field'])}`")->implode(',')) : '';

        return <<<SQL
            INSERT INTO supplier_products (product_id, supplier_id, is_default, created_at, updated_at {$supplierProductColumns})
            SELECT product_id, supplier_id, 1, NOW(), NOW() {$listingsColumns} FROM {$this->temporaryProductsTable} WHERE supplier_id IS NOT NULL;
            SQL;
    }

    private function createTemporaryTagsTable(): string
    {
        $collate = config('database.connections.mysql.collation');

        return <<<SQL
                CREATE TEMPORARY TABLE IF NOT EXISTS {$this->temporaryTagsTable} (
                    `shopify_product_id` bigint(20) unsigned,
                    `tag` varchar(255),
                    PRIMARY KEY (`shopify_product_id`, `tag`)
                ) ENGINE=InnoDB DEFAULT COLLATE={$collate}
                SQL;
    }

    private function setWarehouseToMappedLinesFromPriorityWarehousesQuery(array $priorityWarehouses): string
    {
        $priorityWarehouses = implode(',', $priorityWarehouses);

        return <<<SQL
            UPDATE sales_order_lines sol2 INNER JOIN (
                SELECT sol.id,
                       pi.warehouse_id,
                       ROW_NUMBER() OVER (PARTITION BY sol.id ORDER BY FIELD(pi.warehouse_id, $priorityWarehouses)) priority
                FROM sales_order_lines sol
                    INNER JOIN {$this->temporaryProductsTable} tpi ON tpi.product_listing_id = sol.product_listing_id
                    INNER JOIN products_inventory pi ON sol.product_id = pi.product_id AND pi.warehouse_id IN ($priorityWarehouses) AND pi.inventory_available >= sol.quantity
                WHERE sol.warehouse_id IS null) lpw ON lpw.id = sol2.id AND lpw.priority = 1
            SET sol2.warehouse_id = lpw.warehouse_id
            SQL;
    }

    private function setWarehouseToMappedLinesFromSupplierInventoryQuery(): string
    {
        return <<<SQL
            UPDATE sales_order_lines sol
                INNER JOIN {$this->temporaryProductsTable} tpi ON tpi.product_listing_id = sol.product_listing_id
                INNER JOIN supplier_products sp ON sol.product_id = sp.product_id AND sp.is_default = 1
                INNER JOIN supplier_inventory si ON sol.product_id = si.product_id AND sp.supplier_id = si.supplier_id
                INNER JOIN warehouses w ON w.id = si.warehouse_id AND w.supplier_id IS NOT null AND w.dropship_enabled = 1
            SET sol.warehouse_id = si.warehouse_id
            WHERE sol.warehouse_id IS null AND (si.in_stock = 1 OR si.quantity >= sol.quantity)
            SQL;
    }

    private function setWarehouseToMappedLinesFromFirstPriorityWarehouseQuery(array $priorityWarehouses): string
    {
        $firstPriorityWarehouse = $priorityWarehouses[0];

        return <<<SQL
            UPDATE sales_order_lines sol
                INNER JOIN {$this->temporaryProductsTable} tpi ON tpi.product_listing_id = sol.product_listing_id
            SET sol.warehouse_id = $firstPriorityWarehouse
            WHERE sol.warehouse_id IS null
        SQL;
    }

    private function temporaryColumnName(string $listingField)
    {
        return str_replace('.', '-', $listingField);
    }

    private function insertBrandsDataToTemporaryTable(): void
    {
        if ($brandMapping = $this->mappings->firstWhere('sku_field', 'brand')) {
            $brandColumn = $this->temporaryColumnName($brandMapping['listing_field']);

            $insertQuery = <<<SQL
                INSERT IGNORE INTO `product_brands` (`name`, `created_at`, `updated_at`) 
                SELECT DISTINCT `{$brandColumn}`, NOW(), NOW() FROM `{$this->temporaryProductsTable}` tp 
                WHERE tp.product_exists = 0
                SQL;

            $lookupQuery = <<<SQL
                UPDATE {$this->temporaryProductsTable} AS tp 
                    INNER JOIN product_brands AS pb ON pb.name = `tp`.`{$brandColumn}` 
                SET tp.brand_id = pb.id;
               SQL;

            DB::insert($insertQuery);
            DB::update($lookupQuery);
        }
    }

    private function insertProducts(): void
    {
        DB::insert($this->insertProductsQuery());

        $skuColumn = $this->temporaryColumnName($this->mappings->firstWhere('sku_field', 'sku')['listing_field']);
        $lookupQuery = "UPDATE {$this->temporaryProductsTable} tp INNER JOIN products AS p ON p.sku = tp.{$skuColumn} SET tp.product_id = p.id;";
        DB::statement($lookupQuery);
    }

    private function addDefaultSupplierToProducts(?SupplierPricingTier $supplierPricingTier, ?string $supplierPricingListingField): void
    {
        // default supplier is not mapped
        $defaultSupplierMapping = $this->mappings->firstWhere('sku_field', 'default_supplier');
        if (! $defaultSupplierMapping) {
            return;
        }

        $supplierColumn = $this->temporaryColumnName($defaultSupplierMapping['listing_field']);

        $lookupSuppliersQuery = <<<SQL
            UPDATE {$this->temporaryProductsTable} tp INNER JOIN suppliers s ON s.name = `tp`.`{$supplierColumn}` 
            SET tp.supplier_id = s.id, tp.supplier_warehouse_id = s.default_warehouse_id 
            WHERE tp.product_exists = 0;
           SQL;

        DB::update($lookupSuppliersQuery);

        // add non-existent suppliers
        $suppliersToCreateQuery = "SELECT DISTINCT `{$supplierColumn}` as vendor FROM {$this->temporaryProductsTable} WHERE supplier_id IS NULL AND product_exists = 0";
        $suppliers = collect(DB::select($suppliersToCreateQuery));
        $warehouses = app()->make(WarehouseRepository::class);
        foreach ($suppliers as $supplier) {
            if (empty($supplier->vendor)) {
                continue;
            }

            $supplier = Supplier::query()->create(['name' => $supplier->vendor]);

            if ($supplierPricingTier) {
                $supplier->pricingTiers()->attach($supplierPricingTier);
            }

            $supplier->updateOrCreateAddress([]);
            $warehouses->createDefaultWarehouseForSupplier($supplier);
        }

        DB::update($lookupSuppliersQuery);

        // add supplier products
        $lookupSupplierProductsQuery = <<<SQL
            UPDATE {$this->temporaryProductsTable} tp 
                INNER JOIN supplier_products sp ON sp.supplier_id = tp.supplier_id AND sp.product_id = tp.product_id 
            SET tp.supplier_product_id = sp.id WHERE tp.supplier_id IS NOT NULL;
           SQL;
        DB::insert($this->insertSupplierProductQuery());
        DB::update($lookupSupplierProductsQuery);

        // add supplier product pricing
        if ($supplierPricingTier && $supplierPricingListingField) {
            $insertSupplierProductPricingQuery = <<<SQL
                INSERT INTO supplier_product_pricing (supplier_product_id, supplier_pricing_tier_id, price, created_at, updated_at) 
                SELECT supplier_product_id, {$supplierPricingTier->id}, `{$this->temporaryColumnName($supplierPricingListingField)}`, NOW(), NOW() 
                FROM {$this->temporaryProductsTable} WHERE supplier_product_id IS NOT NULL;
               SQL;

            DB::insert($insertSupplierProductPricingQuery);
        }

        // initialize inventory to supplier products
        $insertSupplierInventoryQuery = <<<SQL
            INSERT INTO supplier_inventory (product_id, supplier_id, warehouse_id, quantity, in_stock, created_at, updated_at) 
            SELECT tp.product_id, tp.supplier_id, tp.supplier_warehouse_id, IF(s.default_stock_level = ?, 0, NULL), IF(s.default_stock_level = ?, 0, 1), NOW(), NOW() FROM {$this->temporaryProductsTable} tp
                INNER JOIN suppliers s ON s.id = tp.supplier_id;
           SQL;
        DB::insert($insertSupplierInventoryQuery, [Supplier::STOCK_LEVEL_ZERO, Supplier::STOCK_LEVEL_NOT_IN_STOCK]);
    }

    private function addPricingToProducts(): void
    {
        $pricingMappings = $this->mappings
            ->filter(fn ($m) => preg_match("/price\..*\.value/", $m['sku_field']))
            ->map(fn ($m) => ['name' => explode('.', $m['sku_field'])[1], 'listing_field' => $m['listing_field']])
            ->keyBy('name');

        // there are no mappings to pricing
        if ($pricingMappings->isEmpty()) {
            return;
        }

        ProductPricingTier::query()
            ->whereIn('name', $pricingMappings->keys())
            ->each(function (ProductPricingTier $pricingTier) use ($pricingMappings) {
                $listingColumn = $this->temporaryColumnName($pricingMappings[$pricingTier->name]['listing_field']);
                $insertProductPricingQuery = <<<SQL
                                INSERT INTO product_pricing (product_pricing_tier_id, product_id, price, created_at, updated_at) 
                                SELECT {$pricingTier->id}, product_id, `{$listingColumn}`, NOW(), NOW() FROM {$this->temporaryProductsTable} 
                                WHERE product_exists = 0 AND `{$listingColumn}` IS NOT NULL;
                               SQL;

                DB::insert($insertProductPricingQuery);
            });
    }

    private function addAttributesToProducts(): void
    {
        $attributeMappings = $this->mappings
            ->filter(fn ($m) => Str::startsWith($m['sku_field'], 'attributes.'))
            ->map(fn ($m) => [
                'name' => str_replace('attributes.', '', $m['sku_field']),
                'listing_field' => $m['listing_field'],
            ])->keyBy('name');

        // there are no mappings to attributes
        if ($attributeMappings->isEmpty()) {
            return;
        }

        Attribute::query()
            ->whereIn('name', $attributeMappings->keys())
            ->each(function (Attribute $attribute) use ($attributeMappings) {
                $listingColumn = $this->temporaryColumnName($attributeMappings[$attribute->name]['listing_field']);
                $insertProductAttributesQuery = <<<SQL
                        INSERT INTO product_attributes (product_id, attribute_id, `value`, created_at, updated_at) 
                        SELECT product_id, {$attribute->id}, `{$listingColumn}`, NOW(), NOW() FROM {$this->temporaryProductsTable} 
                        WHERE product_exists = 0 AND `{$listingColumn}` IS NOT NULL;
                       SQL;

                DB::insert($insertProductAttributesQuery);
            });
    }

    private function addTagsToProducts(array $dataInsertTags): void
    {
        if (empty($dataInsertTags)) {
            return;
        }

        // create tags temporary table
        DB::statement($this->createTemporaryTagsTable());
        // DO NOT use truncate(or drop table), it causes a transaction exception
        DB::delete("DELETE FROM {$this->temporaryTagsTable}");
        // insert tags data to the temporary table
        $columns = $this->columnize(array_keys(reset($dataInsertTags)));
        $values = $this->parameterize($dataInsertTags);
        DB::insert("INSERT IGNORE INTO {$this->temporaryTagsTable} ($columns) VALUES $values");
        // insert tags
        DB::insert("INSERT IGNORE INTO `tags` (`name`) SELECT DISTINCT `tag` FROM `{$this->temporaryTagsTable}`");

        // add tags to products
        $productMorphClass = str_replace('\\', '\\\\', (new \App\Models\Product())->getMorphClass());
        $addTagsToProductsQuery = <<<SQL
            INSERT IGNORE INTO `taggables` (tag_id, taggable_id, taggable_type, created_at, updated_at) 
            SELECT t.id, tp.product_id, '{$productMorphClass}', NOW(), NOW() FROM {$this->temporaryTagsTable} tt 
                INNER JOIN `tags` t ON tt.tag = t.name 
                INNER JOIN {$this->temporaryProductsTable} tp ON tp.shopify_product_id = tt.shopify_product_id AND tp.product_exists = 0
           SQL;
        DB::insert($addTagsToProductsQuery);
    }

    private function addMainCategoryToProducts(): void
    {
        // category_main is not mapped
        if (! $categoryMapping = $this->mappings->firstWhere('sku_field', 'category_main')) {
            return;
        }

        $listingColumn = $this->temporaryColumnName($categoryMapping['listing_field']);
        $lookupCategoriesQuery = <<<SQL
            UPDATE {$this->temporaryProductsTable} tp 
                INNER JOIN product_categories pc ON pc.name = `tp`.`{$listingColumn}` AND pc.parent_id IS NULL 
            SET tp.category_id = pc.id WHERE tp.category_id IS NULL;
           SQL;
        DB::update($lookupCategoriesQuery);

        // insert the non-existent categories
        $insertProductCategoriesQuery = <<<SQL
            INSERT INTO `product_categories` (name, is_leaf, path, level, created_at, updated_at) 
            SELECT `tp`.`{$listingColumn}`, 1, `tp`.`{$listingColumn}`, 1, NOW(), NOW() 
            FROM {$this->temporaryProductsTable} tp WHERE tp.category_id IS NULL AND product_exists = 0;
           SQL;
        DB::insert($insertProductCategoriesQuery);
        DB::update($lookupCategoriesQuery);

        // add categories to products
        $addCategoriesToProductsQuery = <<<SQL
            INSERT INTO product_to_categories (product_id, category_id, is_primary, created_at, updated_at)
            SELECT tp.product_id, tp.category_id, 1, NOW(), NOW() 
            FROM {$this->temporaryProductsTable} tp WHERE tp.category_id IS NOT NULL AND product_exists = 0;
           SQL;
        DB::insert($addCategoriesToProductsQuery);

        // add attributes to products from category's attribute groups
        $addAttributesFromAttributeGroupsQuery = <<<SQL
            INSERT INTO product_attributes (product_id, attribute_id, attribute_group_id, created_at, updated_at) 
            SELECT tp.product_id, att.id, att.attribute_group_id, NOW(), NOW() FROM {$this->temporaryProductsTable} tp 
                INNER JOIN categories_to_attribute_groups ctag ON ctag.category_id = tp.category_id 
                INNER JOIN attributes att ON att.attribute_group_id = ctag.attribute_group_id 
            WHERE tp.category_id IS NOT NULL AND tp.product_exists = 0
            ON DUPLICATE KEY UPDATE `attribute_group_id` = VALUES(`attribute_group_id`)
           SQL;
        DB::insert($addAttributesFromAttributeGroupsQuery);

        // add category's attributes to products
        $addCategoryAttributesToProducts = <<<SQL
            INSERT IGNORE INTO product_attributes (product_id, attribute_id, created_at, updated_at) 
            SELECT tp.product_id, cta.attribute_id, NOW(), NOW() FROM {$this->temporaryProductsTable} tp 
                INNER JOIN categories_to_attributes cta ON cta.category_id = tp.category_id 
            WHERE tp.category_id IS NOT NULL AND tp.product_exists = 0;
           SQL;
        DB::insert($addCategoryAttributesToProducts);
    }

    private function insertProductListings(): void
    {
        $shopifyProductMorphClass = str_replace('\\', '\\\\', (new ShopifyProduct())->getMorphClass());

        $inventoryQuantity = 'null';
        if (ShopifyWebhook::hasInventoryLevelUpdateWebhook($this->integrationInstance->id)) {
            $inventoryQuantity = 'inventory_quantity';
        }

        $InsertProductListingsQuery = <<<SQL
              INSERT INTO product_listings (`product_id`, `sales_channel_id`, `sales_channel_listing_id`, `listing_sku`, `title`, `price`, `quantity`, `sales_channel_qty`, `sales_channel_qty_last_updated`, `document_id`, `created_at`, `updated_at`, `document_type`) 
              SELECT product_id, {$this->integrationInstance->salesChannel->id}, variant_id, shopify_product_sku, shopify_product_title, shopify_product_price, 0, {$inventoryQuantity}, NOW(), shopify_product_id, NOW(), NOW(), '{$shopifyProductMorphClass}'
              FROM {$this->temporaryProductsTable} ON DUPLICATE KEY 
                  UPDATE `quantity` = `quantity`, `sales_channel_qty` = VALUES(`sales_channel_qty`), `sales_channel_qty_last_updated` = VALUES(`sales_channel_qty_last_updated`), `document_id` = VALUES(`document_id`), `updated_at` = VALUES(`updated_at`), `document_type` = VALUES(`document_type`  );
           SQL;
        DB::insert($InsertProductListingsQuery);

        $lookupProductListingsQuery = <<<SQL
            UPDATE {$this->temporaryProductsTable} tp 
                INNER JOIN product_listings pl ON pl.sales_channel_listing_id = tp.variant_id AND pl.sales_channel_id = {$this->integrationInstance->salesChannel->id} 
                INNER JOIN shopify_products sp ON sp.id = tp.shopify_product_id 
            SET tp.product_listing_id = pl.id, sp.product = pl.id, sp.mapped_at = NOW();
           SQL;
        DB::update($lookupProductListingsQuery);
    }

    private function mapSalesOrderLines(): void
    {
        $priorityWarehouses = json_decode(Helpers::setting(Setting::KEY_WAREHOUSE_PRIORITY), true);
        $setFirstWarehouseQuery = '';
        if (empty($priorityWarehouses)) {
            $setFirstWarehouseQuery = ', sol.warehouse_id = '.Warehouse::with([])->value('id');
        }

        $mapSalesOrderLinesQuery = <<<SQL
            UPDATE sales_order_lines sol 
                INNER JOIN shopify_order_line_items soli ON soli.line_id = sol.sales_channel_line_id 
                INNER JOIN {$this->temporaryProductsTable} tp ON tp.variant_id = soli.variant_id 
            SET sol.product_listing_id = tp.product_listing_id, sol.product_id = tp.product_id $setFirstWarehouseQuery 
            WHERE sol.sales_channel_line_id REGEXP "^([0-9])+$";
           SQL;
        DB::update($mapSalesOrderLinesQuery);

        // set the warehouse_id to the mapped lines
        if (! empty($priorityWarehouses)) {
            // actually, these queries for existing skus
            DB::statement($this->setWarehouseToMappedLinesFromPriorityWarehousesQuery($priorityWarehouses));
            DB::statement($this->setWarehouseToMappedLinesFromSupplierInventoryQuery());
            // this query that will actually be used to new products, because the product does not have an inventory
            DB::statement($this->setWarehouseToMappedLinesFromFirstPriorityWarehouseQuery($priorityWarehouses));
        }

        // handle sales orders after map lines
        $salesOrderLines = SalesOrderLine::with(['product', 'salesOrder.shopifyOrder', 'salesOrder.salesOrderLines'])
            ->join($this->temporaryProductsTable, "{$this->temporaryProductsTable}.product_listing_id", '=', 'sales_order_lines.product_listing_id');

        $salesOrderLines->each(function (SalesOrderLine $salesOrderLine) {
            echo 'Processing sales order line '.$salesOrderLine->product->sku.' for '.$salesOrderLine->salesOrder->sales_order_number."\n";
            ShopifyMapSalesOrderLinesAfterCreateSkuProductsJob::dispatch(
                $salesOrderLine->salesOrder->id,
                $salesOrderLine->salesOrder->shopifyOrder->id
            );
        });
    }

    private function columnize($columns): string
    {
        return implode(', ', array_map(fn ($k) => "`$k`", $columns));
    }

    private function parameterize($data): string
    {
        return collect($data)
            ->map(function ($record) {
                $values = array_map(function ($v) {
                    return is_null($v) ? 'null' : ("'".str_replace("'", "\'", stripslashes($v))."'");
                }, array_values($record));

                return '('.implode(',', $values).')';
            })
            ->implode(', ');
    }
}
