<?php

namespace App\Jobs\Shopify;

use App\Models\IntegrationInstance;
use App\Models\Shopify\ShopifyOrder;
use Illuminate\Bus\Batchable;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldBeUnique;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Throwable;

class GenerateCreateSalesOrderFromShopifyOrderJobJobs implements ShouldBeUnique, ShouldQueue
{
    use Batchable,Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public int $uniqueFor = 60 * 10;

    public function uniqueId(): string
    {
        return 'generate-create-sales-order-from-shopify-order-jobs:'.$this->integrationInstance->id.'-'.implode(',', $this->ids);
    }

    /**
     * Create a new job instance.
     */
    public function __construct(protected IntegrationInstance|int $integrationInstance, protected array $ids = [])
    {
        $this->onQueue('salesOrderProcessing');
        $this->integrationInstance = $integrationInstance instanceof IntegrationInstance ? $integrationInstance : IntegrationInstance::query()->findOrFail($integrationInstance);
    }

    /**
     * Execute the job.
     *
     * @throws Throwable
     */
    public function handle(): void
    {
        if (! empty($this->ids)) {
            customlog('shopifyOrdersBenchmark', 'GenerateCreateSalesOrderFromShopifyOrderJobJobs (manual) processing started for '.count($this->ids).' orders');
            foreach (ShopifyOrder::getOrderNamesFromIds($this->integrationInstance, $this->ids) as $orderName) {
                CreateSalesOrderFromShopifyOrderJob::dispatch($this->integrationInstance, $orderName)->onQueue('salesOrderProcessing');
            }
        } else {
            customlog('shopifyOrdersBenchmark', 'GenerateCreateSalesOrderFromShopifyOrderJobJobs (auto) processing started for ' . $this->integrationInstance->name . ' integration instance');
            $ordersToCreate = ShopifyOrder::getUnprocessedOrderNames($this->integrationInstance);
            customlog('shopifyOrdersBenchmark', count($ordersToCreate).' orders to create for ' . $this->integrationInstance->name . ' integration instance');
            foreach ($ordersToCreate as $orderName) {
                CreateSalesOrderFromShopifyOrderJob::dispatch($this->integrationInstance, $orderName)->onQueue('salesOrderProcessing');
            }
            customlog('shopifyOrdersBenchmark', 'Jobs for creating orders dispatched');
        }
    }
}
