<?php

namespace App\Jobs\ShipStation;

use App\Integrations\ShipStation;
use App\Models\IntegrationInstance;
use App\Models\ShippingMethodMappingsSkuToShippingProviderMethod;
use App\SDKs\ShipStation\ShipStationException;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

class GetShippingServices implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * @var IntegrationInstance
     */
    protected $integrationInstance;

    /**
     * The processing job summary.
     *
     * @var array
     */
    protected $summary = ['Carriers Count' => 0, 'Services Count' => 0];

    /**
     * Create a new job instance.
     */
    public function __construct(IntegrationInstance $integrationInstance)
    {
        $this->integrationInstance = $integrationInstance;
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        set_time_limit(0);

        $shipStation = new ShipStation($this->integrationInstance);

        $existsServices = [];
        try {
            foreach ($shipStation->getCarriers() as $carrier) {
                foreach ($shipStation->getShippingServices($carrier['code']) as $service) {
                    $shippingServiceMapping = ShippingMethodMappingsSkuToShippingProviderMethod::query()->firstOrCreate(
                        [
                            'shipping_provider_id' => $this->integrationInstance->id,
                            'shipping_provider_carrier' => $service['carrierCode'],
                            'shipping_provider_method' => $service['code'],
                        ]
                    );

                    $existsServices[] = $shippingServiceMapping->id;

                    // SUMMARY
                    $this->summary['Services Count'] += 1;
                }

                // SUMMARY
                $this->summary['Carriers Count'] += 1;
            }
        } catch (ShipStationException $shipStationException) {
            // 401 Unauthorized
            if ($shipStationException->getCode() == 401) {
                $this->integrationInstance->unauthorizedConnection();
            }
            throw $shipStationException;
        }

        // delete services that not retrieve from ShipStation
        ShippingMethodMappingsSkuToShippingProviderMethod::query()
            ->where('shipping_provider_id', $this->integrationInstance->id)
            ->whereNotIn('id', $existsServices)
            ->delete();

        // for logging
        $this->options = ['Integration Instance' => $this->integrationInstance->name.' ('.$this->integrationInstance->id.') '];
    }
}
