<?php

namespace App\Jobs\ShipStation;

use App\Exceptions\ShipStationGatewayTimeoutException;
use App\Integrations\ShipStation;
use App\Models\IntegrationInstance;
use App\Models\SalesOrderFulfillment;
use App\Models\ShippingMethodMappingsSkuToShippingProviderMethod;
use App\Models\ShipStation\ShipstationOrder;
use App\Notifications\MonitoringMessage;
use App\SDKs\ShipStation\Model\Shipment;
use App\SDKs\ShipStation\Requests\ListShipmentsRequest;
use App\SDKs\ShipStation\ShipStationException;
use App\Services\SalesOrder\FulfillSalesOrderService;
use App\Services\Shopify\ShopifySubmitTrackingInfo;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Notification;

class GetShipments implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * @var IntegrationInstance
     */
    protected $integrationInstance;

    protected $listShipmentsRequest;

    protected $summary = ['success' => 0, 'fails' => 0, 'errors' => []];

    /**
     * Create a new job instance.
     */
    public function __construct(IntegrationInstance $integrationInstance, ?ListShipmentsRequest $listShipmentsRequest = null)
    {
        $this->integrationInstance = $integrationInstance;
        $this->listShipmentsRequest = $listShipmentsRequest ?: new ListShipmentsRequest();
    }

    /**
     * Execute the job.
     *
     * @throws ShipStationException
     */
    public function handle(): void
    {
        set_time_limit(0);

        $shipStation = app()->makeWith(ShipStation::class, ['integrationInstance' => $this->integrationInstance]);

        try {
            $shipments = $shipStation->getShipments($this->listShipmentsRequest);
            foreach ($shipments as $shipmentResponse) {
                if ($shipmentResponse instanceof ListShipmentsRequest) {
                    $this->listShipmentsRequest = $shipmentResponse;

                    continue;
                }

                $shippedOrders = collect($shipmentResponse->shipments)->map(function ($shipment) {
                    return $shipment->toArray();
                });
                if ($shippedOrders->isEmpty()) {
                    return;
                }

                ShipstationOrder::with('salesOrderFulfillment.salesOrder')
                    ->whereIn('orderId', $shippedOrders->pluck('orderId')->toArray())
                    ->each(function (ShipstationOrder $order) use ($shippedOrders) {
                        $shipment = new Shipment($shippedOrders->firstWhere('orderId', $order->orderId));

                        // get/create shipping method mapping
                        $shippingProviderMethodMap = ShippingMethodMappingsSkuToShippingProviderMethod::query()->firstOrCreate([
                            'shipping_provider_id' => $this->integrationInstance->id,
                            'shipping_provider_carrier' => $shipment->carrierCode,
                            'shipping_provider_method' => $shipment->serviceCode, ]);

                        if (! $order->salesOrderFulfillment) {
                            return; // continue
                        }

                        $salesOrderFulfillment = $order->salesOrderFulfillment;
                        // update sales order fulfillment
                        $salesOrderFulfillment->tracking_number = $shipment->trackingNumber ?: $salesOrderFulfillment->tracking_number;
                        $salesOrderFulfillment->cost = $shipment->shipmentCost + $shipment->insuranceCost;
                        $salesOrderFulfillment->fulfilled_shipping_method_id = $shippingProviderMethodMap->shipping_method_id;
                        $salesOrderFulfillment->fulfilled_shipping_method = $shipment->carrierCode;
                        $salesOrderFulfillment->fulfilled_at = $shipment->shipDate ?: $shipment->createDate;
                        $salesOrderFulfillment->status = SalesOrderFulfillment::STATUS_FULFILLED;
                        $salesOrderFulfillment->save();

                        // submit tracking to the sales channel
                        try {
                            $response = ShopifySubmitTrackingInfo::factory($salesOrderFulfillment)?->submit();
                            if (isset($response['success']) && ! $response['success']) {
                                $errorMessage = "Can't submit tracking to the sales channel: ".$salesOrderFulfillment->salesOrder->sales_order_number.', response: '.json_encode($response);
                                Notification::route('slack', config('slack.debugging'))->notify(new MonitoringMessage($errorMessage));
                                Log::debug($errorMessage);
                            }
                        } catch (\Exception $e) {
                            $errorMessage = "Can't submit tracking to the sales channel: ".$salesOrderFulfillment->salesOrder->sales_order_number.', exception: '.$e->getMessage();
                            Notification::route('slack', config('slack.debugging'))->notify(new MonitoringMessage($errorMessage));
                            Log::debug($errorMessage, $e->getTrace());
                        }

                        // mark sales order as fulfilled
                        FulfillSalesOrderService::make($salesOrderFulfillment->salesOrder)->updateFulfillmentStatus($salesOrderFulfillment->fulfilled_at);

                        // add the shipment to the Shipstation order
                        $order->shipment = $shipment->toArray() + ['isWebhook' => false];
                        $order->save();
                    });
            }
        } catch (ShipStationException $shipStationException) {
            // 429 Too Many Requests
            if ($shipStationException->getCode() == 500 || $shipStationException->getCode() == 429) {
                // dispatch the job again after 2 minutes
                dispatch(new static($this->integrationInstance, $this->listShipmentsRequest))
                    ->delay(now()->addMinutes(2)) // delay 2 minutes
                    ->onQueue($this->queue)
                    ->onConnection($this->connection);

                return;
            }

            // 401 Unauthorized
            if ($shipStationException->getCode() == 401) {
                customlog('SKU-6888', 'Shipstation 401 error', [
                    'message' => $shipStationException->getMessage(),
                    'code' => $shipStationException->getCode(),
                ], 7);
                $this->integrationInstance->unauthorizedConnection();
            }

            throw $shipStationException;
        } catch (ShipStationGatewayTimeoutException) {
            dispatch(new static($this->integrationInstance, $this->listShipmentsRequest))
                ->delay(now()->addMinutes(2)) // delay 2 minutes
                ->onQueue($this->queue)
                ->onConnection($this->connection);

            return;
        }

        // for logging
        $this->options = $this->listShipmentsRequest->toArray();
        $this->options['Integration Instance'] = $this->integrationInstance->name.' ('.$this->integrationInstance->id.') ';
    }

    public function setRequestOptions(array $options)
    {
        $this->listShipmentsRequest->setAttributes($options);
    }
}
