<?php

namespace App\Jobs;

use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\Integrations\SalesChannelOrder;
use App\Models\IntegrationInstance;
use App\Models\ProductListing;
use App\Models\SalesOrder;
use App\Models\Shopify\ShopifyOrder;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Collection;
use Throwable;

/**
 * @deprecated This is the old sales order line mapper which should only still be used by Shopify.  This is replaced
 *  by MapSalesOrderLinesToSalesChannelProductsJob
 */
abstract class SalesOrderLineMapper implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    const STATUS_NEW = 'new';

    const STATUS_UPDATED = 'updated';

    const STATUS_DELETED = 'deleted';

    /**
     * @var IntegrationInstance
     */
    protected $integrationInstance;

    protected $productListing;

    /**
     * Status of product listing, values in: new, updated and deleted.
     */
    protected $status;

    /**
     * Create a new job instance.
     *
     * @param $productListingSku
     * @param $productListingId
     * @param $productId
     */
    public function __construct(IntegrationInstance|IntegrationInstanceInterface $integrationInstance, ProductListing $productListing, $status)
    {
        $this->integrationInstance = $integrationInstance;
        $this->productListing = $productListing;
        $this->status = $status;
    }

    /**
     * Execute the job.
     *
     * @throws Throwable
     */
    public function handle(): void
    {
        /** @see SKU-4215 it should only affect future orders. */
        if ($this->status == self::STATUS_DELETED) {
            return;
        }

        set_time_limit(0);
        $orders = $this->getOrders();
        if ($orders->isEmpty()) {
            return;
        }

        /** @var ShopifyOrder $order */
        $order = $orders->first();
        $salesOrderIdColumnName = $order->salesOrder()->getForeignKeyName();

        $salesOrders = SalesOrder::with(['salesOrderLines'])
            ->whereIn('id', $orders->pluck($salesOrderIdColumnName)->toArray())
            ->get();

        /** @var SalesOrder $salesOrder */
        foreach ($salesOrders as $salesOrder) {
            /** @var SalesChannelOrder $order */
            $order = $orders->firstWhere($salesOrderIdColumnName, $salesOrder->id);

            $skuOrderLines = $order->getSkuOrderLines();
            // This splits bundle components if needed.
            $salesOrder->setSalesOrderLines($skuOrderLines, true);

            // map sales order line
            $items = $order->findMatchingLines(collect($skuOrderLines), $this->productListing);
            if (! empty($items)) {
                foreach ($items as $item) {
                    $matchingLine = $salesOrder->salesOrderLines->firstWhere('sales_channel_line_id', $item['sales_channel_line_id']);
                    $matchingLine?->update([
                        'product_id' => $this->status != self::STATUS_DELETED ? $this->productListing->product_id : null,
                        'warehouse_id' => $this->status != self::STATUS_DELETED ? $item['warehouse_id'] : null,
                        'product_listing_id' => $this->status != self::STATUS_DELETED ? $this->productListing->id : null,
                    ]);
                }
            } else {
                // the listing mapped with bundle product and the line mapped to bundle components
                $salesOrder->setSalesOrderLines($skuOrderLines, true);
            }

            $order->updateSKUOrder();
        }
    }

    /**
     * @return Collection
     */
    abstract protected function getOrders();
}
