<?php

namespace App\Jobs;

use App\Models\Address;
use App\Models\Addressable;
use App\Models\Customer;
use App\Models\PurchaseOrder;
use App\Models\SalesCredit;
use App\Models\SalesOrder;
use App\Models\Store;
use App\Models\Supplier;
use App\Models\Warehouse;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldBeUnique;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Throwable;

class PurgeUnusedCustomersAndAddresses implements ShouldBeUnique, ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public int $uniqueFor = 60 * 15;

    /**
     * Create a new job instance.
     *
     * @return void
     */
    public function __construct()
    {
    }

    /**
     * Execute the job.
     *
     * @throws Throwable
     */
    public function handle(): void
    {
        DB::transaction(function () {
            $customers = Customer::query()
                ->whereDoesntHave('salesOrders')
                ->whereDoesntHave('salesCredits');

            //Log::info('Purging ' . $customers->count() . ' total unused customers');

            $customers->chunk(1000, function ($chunk) {
                $addressables = Addressable::query()
                    ->whereIn('addressable_id', $chunk->pluck('id'))
                    ->where('addressable_type', Customer::class);
                //Log::info('Purging ' . $addressables->count() . ' customer addressables');
                $addressables->delete();

                //Log::info('Purging ' . $chunk->count() . ' unused customers');
                Customer::query()->whereIn('id', $chunk->pluck('id'))->delete();
            });

            $usedAddressIdsQuery = Addressable::query()->whereNotNull('address_id')->select('address_id as id')
                ->union(Customer::query()->whereNotNull('default_billing_address_id')->select('default_billing_address_id as id'))
                ->union(Customer::query()->whereNotNull('default_shipping_address_id')->select('default_shipping_address_id as id'))
                ->union(Supplier::query()->whereNotNull('address_id')->select('address_id as id'))
                ->union(Warehouse::query()->whereNotNull('address_id')->select('address_id as id'))
                ->union(PurchaseOrder::query()->whereNotNull('destination_address_id')->select('destination_address_id as id'))
                ->union(Store::query()->whereNotNull('address_id')->select('address_id as id'))
                ->union(SalesOrder::query()->whereNotNull('shipping_address_id')->select('shipping_address_id as id'))
                ->union(SalesOrder::query()->whereNotNull('billing_address_id')->select('billing_address_id as id'))
                ->union(SalesCredit::query()->whereNotNull('from_address_id')->select('from_address_id as id'));

            // Delete unused addresses
            $addresses = Address::query()
                ->leftJoinSub($usedAddressIdsQuery, 'used_addresses', 'used_addresses.id', '=', 'addresses.id')
                ->whereNull('used_addresses.id');
            $addresses->delete();
        });
    }
}
