<?php

namespace App\Jobs;

use App\Helpers\ExcelHelper;
use App\Models\PurchaseOrder;
use App\Models\Supplier;
use App\Notifications\BulkPurchaseOrderNotification;
use Illuminate\Bus\Batchable;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

/**
 * Class BulkPurchaseOrderEmailJob.
 */
class PurchaseOrderSubmissionBatchingJob implements ShouldQueue
{
    use Batchable;
    use Dispatchable;
    use InteractsWithQueue;
    use Queueable;
    use SerializesModels;

    protected Supplier $supplier;

    /**
     * BulkPurchaseOrderEmailJob constructor.
     */
    public function __construct(Supplier $supplier)
    {
        $this->supplier = $supplier;
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        //set_time_limit(0);

        $attachments = $this->getAttachments();
        if (! empty($attachments)) {
            $this->supplier->notify(new BulkPurchaseOrderNotification($this->supplier, $attachments));
            // Mark the PO's as submitted
            $this->supplier->purchaseOrders()->where('submission_status', PurchaseOrder::SUBMISSION_STATUS_QUEUED)
                ->update(['submission_status' => PurchaseOrder::SUBMISSION_STATUS_SUBMITTED]);
        }
    }

    private function getAttachments(): array
    {
        return array_merge(
            $this->makeCSVAttachment(),
            $this->getPDFAttachments()
        );
    }

    private function makeCSVAttachment(): array
    {
        $records = [];
        $this->supplier
            ->purchaseOrders()
            ->where('submission_status', PurchaseOrder::SUBMISSION_STATUS_QUEUED)
            ->where(function (Builder $builder) {
                $builder->where('submission_format', PurchaseOrder::SUBMISSION_FORMAT_CSV_ATTACHMENT)
                    ->orWhere('submission_format', PurchaseOrder::SUBMISSION_FORMAT_EMAIL_PDF_AND_CSV_ATTACHMENTS);
            })->get()->each(function (PurchaseOrder $order) use (&$records) {
                $lines = $order->linesForCSV();
                foreach ($lines as $key => $line) {
                    $line['PO #'] = $order->purchase_order_number;
                    $records[] = $line;
                }
            });

        if (empty($records)) {
            return [];
        }

        $headers = array_keys($records[0]);

        return [ExcelHelper::array2csvfile('invoice.csv', $headers, $records)];
    }

    private function getPDFAttachments(): array
    {
        return $this->supplier
            ->purchaseOrders()
            ->where('submission_status', PurchaseOrder::SUBMISSION_STATUS_QUEUED)
            ->where('submission_format', PurchaseOrder::SUBMISSION_FORMAT_PDF_ATTACHMENT)
            ->get()->map(function (PurchaseOrder $order) {
                return $order->exportToPDF();
            })->toArray();
    }
}
