<?php

namespace App\Jobs;

use App\Models\PackingSlipQueue;
use App\Notifications\MonitoringMessage;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Facades\Redis;

class PackingSlipHandlerJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public const RESTART_KEY = 'sku:packing_slip_handler_restart';

    const MAX_PROCESSING_COUNT = 25;

    private int $currentProcessingCount = 0;

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        set_time_limit(0);

        $startedTime = now()->getTimestamp();

        while (true) {
            try {
                if ($this->shouldRestart($startedTime)) {
                    return;
                }

                $this->dispatchPackingSlips();
                $this->resetProcessing();
            } catch (\Throwable $exception) {
                $this->printError($exception->getMessage());
                //Log::error($exception->getMessage().": \n".$exception->getTraceAsString());
            }

            sleep(20);
        }
    }

    private function dispatchPackingSlips()
    {
        // current running reports
        $this->currentProcessingCount = PackingSlipQueue::processing()->count();
        if ($this->currentProcessingCount >= self::MAX_PROCESSING_COUNT) {
            return;
        }

        PackingSlipQueue::pending()
            ->limit(self::MAX_PROCESSING_COUNT - $this->currentProcessingCount)
            ->get()
            ->each(function (PackingSlipQueue $packingSlipQueue) {
                try {
                    $packingSlipQueue->dispatch();
                    printf("[%s] %s \n", date('Y-m-d H:i:s'), "Dispatched {$packingSlipQueue->link_type}:{$packingSlipQueue->link_id}");
                } catch (\Throwable $exception) {
                    $this->printError("Failed {$packingSlipQueue->link_type}:{$packingSlipQueue->link_id} {$exception->getMessage()}");
                }
            });
    }

    private function resetProcessing()
    {
        PackingSlipQueue::processing()
            ->where('queued_at', '<', now()->subMinutes(30))
            ->update(['status' => PackingSlipQueue::STATUS_PENDING]);
    }

    private function printError(string $message)
    {
        printf("[%s] [Error]: %s \n", date('Y-m-d H:i:s'), $message);
        // send a Slack notification
        //Notification::route('slack', config('slack.debugging'))->notify(new MonitoringMessage("PackingSlipHandler: " . $message));
    }

    private function shouldRestart($startTime): bool
    {
        $lastRestartTime = Redis::connection()->get(self::RESTART_KEY);

        return $lastRestartTime >= $startTime;
    }
}
