<?php

namespace App\Jobs;

use App\Models\BackorderQueue;
use App\Models\BackorderQueueCoverage;
use App\Repositories\BackorderQueueRepository;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

/**
 * Class MoveBackorderCoveragesUp.
 */
class MoveBackorderCoveragesUp implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * The backorder queue we need to make up coverage for.
     *
     * @var BackorderQueue
     */
    protected $backorderQueue;

    /**
     * @var int The quantity to make up for the queue
     */
    protected $quantity;

    /**
     * @var BackorderQueueRepository
     */
    protected $queues;

    /**
     * MoveBackorderCoveragesUp constructor.
     *
     *
     * @throws BindingResolutionException
     */
    public function __construct(int $quantity, BackorderQueue $backorderQueue)
    {
        $this->quantity = $quantity;
        $this->backorderQueue = $backorderQueue;
        $this->queues = app()->make(BackorderQueueRepository::class);
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        if (! $this->isActiveQueue()) {
            return;
        }

        $queues = $this->queues->afterPriority(
            $this->backorderQueue->priority,
            $this->backorderQueue->salesOrderLine->product_id,
            'desc',
            $this->backorderQueue->supplier_id
        );

        $remainingQuantity = $this->quantity;

        foreach ($queues as $queue) {
            /** @var BackorderQueue $queue */
            if ($queue->covered_quantity >= $remainingQuantity) {
                // The current queue's coverage can make up the quantity needed,
                // we simply move them to the priority queue and end.
                $this->transferCoveragesFrom($queue, $remainingQuantity);
                break;
            } else {
                // The queue's coverage isn't enough, we transfer all the coverages
                // and move to the next queue.
                $remainingQuantity -= $queue->covered_quantity;
                $this->transferAllCoveragesFrom($queue);
            }
        }
    }

    protected function transferCoveragesFrom(BackorderQueue $queue, int $quantity)
    {
        if ($quantity == $queue->covered_quantity) {
            $this->transferAllCoveragesFrom($queue);

            return;
        }

        // We iteratively transfer coverages until we use up all the
        // quantity.
        $coverages = $queue->backorderQueueCoverages()
            ->active()
            ->get();
        $remainingQuantity = $quantity;
        foreach ($coverages as $coverage) {
            /** @var BackorderQueueCoverage $coverage */
            if ($coverage->unreleased_quantity >= $remainingQuantity) {
                $this->transferCoverageQuantity($coverage, $remainingQuantity);
                break;
            } else {
                $this->transferAllCoverageQuantity($coverage);
            }
        }
    }

    protected function transferCoverageQuantity(BackorderQueueCoverage $coverage, int $quantity)
    {
        if ($coverage->unreleased_quantity == $quantity) {
            $this->transferAllCoverageQuantity($coverage);

            return;
        }

        // Charge the quantity to the coverage
        $coverage->released_quantity += $quantity;
        $coverage->save();

        // Create coverage for the queue with the quantity.
        $this->queues->createCoverage($quantity, $this->backorderQueue, $coverage->purchaseOrderLine);
    }

    protected function transferAllCoverageQuantity(BackorderQueueCoverage $coverage)
    {
        $coverage->backorder_queue_id = $this->backorderQueue->id;
        $coverage->save();
    }

    protected function transferAllCoveragesFrom(BackorderQueue $queue)
    {
        // We move all coverages from the provided queue to the priority queue.
        $this->queues->moveCoverages($queue, $this->backorderQueue);
    }

    protected function isActiveQueue(): bool
    {
        return ! $this->backorderQueue->is_fully_released
           && ! is_null($this->backorderQueue->priority);
    }
}
