<?php

namespace App\Jobs\Middleware;

use Illuminate\Container\Container;
use Illuminate\Contracts\Cache\Repository as Cache;

class CheckUniqueJobsMiddleware
{
    public function __construct(protected array $jobs)
    {
    }

    public function handle($job, $next): void
    {
        foreach ($this->jobs as $subJob) {
            $lock = $this->getCache($subJob)->lock($this->getLockKey($job));

            if ($lock->get()) {
                // the sub-job was neither queued nor processing, just release the lock
                $lock->release();
            } else {
                // the sub-job is either queued or processing, so don't dispatch the job
                return;
            }
        }

        $next($job);
    }

    /**
     * Get the unique lock key of the job
     */
    private function getLockKey($job): string
    {
        $uniqueId = method_exists($job, 'uniqueId')
          ? $job->uniqueId()
          : ($job->uniqueId ?? '');

        return 'laravel_unique_job:'.get_class($job).$uniqueId;
    }

    /**
     * Get the cache repository
     *
     *
     * @throws \Illuminate\Contracts\Container\BindingResolutionException
     */
    private function getCache($job): \Illuminate\Support\Facades\Cache
    {
        return method_exists($job, 'uniqueVia')
          ? $job->uniqueVia()
          : Container::getInstance()->make(Cache::class);
    }
}
