<?php

namespace App\Jobs\Magento;

use App\Helpers;
use App\Models\IntegrationInstance;
use App\Models\ProductListing;
use App\Models\ProductPricingTier;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Cache\Repository;
use Illuminate\Contracts\Queue\ShouldBeUnique;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Query\JoinClause;
use Illuminate\Database\QueryException;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Throwable;

class SyncSkuPricingJob implements ShouldBeUnique, ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public $tries = 20;

    public int $uniqueFor = 60 * 60;

    /**
     * Create a new job instance.
     *
     * @return void
     */
    public function __construct(protected IntegrationInstance $integrationInstance)
    {
        //
    }

    /**
     * Sync pricing from Magento To SKU
     * Magento is the master of price
     */
    public function handle(): void
    {
        $masterOfPrice = strtolower($this->integrationInstance->integration_settings['pricing']['masterOfPrice']['id']);
        $productPricingTierId = $this->integrationInstance->integration_settings['pricing']['pricingTier']['id'] ??
                                ProductPricingTier::query()->where('is_default', true)->value('id');

        // update the price depends on the pricing tier of the integration instance setting
        // get products that need sync
        $query = ProductListing::query()->where('sales_channel_id', $this->integrationInstance->salesChannel->id);
        // master of price is sales_channel(Magento)
        if ($masterOfPrice == ProductListing::MASTER_SALES_CHANNEL) {
            $query->where(function (Builder $builder) {
                $builder->whereNull('master_of_price');
                $builder->orWhere('master_of_price', ProductListing::MASTER_SALES_CHANNEL);
            });
        } else {
            $query->where('master_of_price', ProductListing::MASTER_SALES_CHANNEL);
        }

        try {
            // update the selected(or the default) pricing tier
            $needUpdateQuery = $query->clone()->join('magento_products', function (JoinClause $join) {
                $join->on('magento_products.product', '=', 'product_listings.id');
            })->leftJoin('product_pricing', function (JoinClause $join) use ($productPricingTierId) {
                $join->on('product_listings.product_id', '=', 'product_pricing.product_id')
                    ->where('product_pricing.product_pricing_tier_id', DB::raw("IFNULL(`product_listings`.`product_pricing_tier_id`, $productPricingTierId)"));
            })->where(function (Builder $builder) {
                $builder->whereColumn('magento_products.price', '!=', 'product_pricing.price');
                $builder->orWhereNull('product_pricing.price');
            })->select([
                DB::raw("IFNULL(`product_listings`.`product_pricing_tier_id`, $productPricingTierId)"),
                'product_listings.product_id',
                'magento_products.price',
            ]);

            $insertQuery = "INSERT INTO `product_pricing` (`product_pricing_tier_id`, `product_id`, `price`) {$needUpdateQuery->toSql()} ON DUPLICATE KEY UPDATE `price`=VALUES(`price`)";
            DB::insert($insertQuery, $needUpdateQuery->getBindings());

            // update other tier prices
            $magentoTierPriceValue = "JSON_EXTRACT(`json_object`, JSON_UNQUOTE(REPLACE(JSON_SEARCH(`json_object`, 'one', `magento_customer_groups`.`customer_group_id`, NULL, '$.tier_prices[*].customer_group_id'), 'customer_group_id', 'value')))";
            $magentoTierPriceQty = "JSON_EXTRACT(`json_object`, JSON_UNQUOTE(REPLACE(JSON_SEARCH(`json_object`, 'one', `magento_customer_groups`.`customer_group_id`, NULL, '$.tier_prices[*].customer_group_id'), 'customer_group_id', 'qty')))";

            $needUpdateQuery = $query->clone()
                ->leftJoin('magento_customer_groups', 'magento_customer_groups.integration_instance_id', '=', DB::raw($this->integrationInstance->id))
                ->join('magento_products', 'magento_products.product', '=', 'product_listings.id')
                ->leftJoin('product_pricing', function (JoinClause $join) {
                    $join->on('product_listings.product_id', '=', 'product_pricing.product_id')
                        ->whereColumn('product_pricing.product_pricing_tier_id', 'magento_customer_groups.product_pricing_tier_id');
                })
                ->whereRaw("$magentoTierPriceValue IS NOT NULL")
                ->whereRaw("$magentoTierPriceQty = 1")
                ->where(function (Builder $builder) use ($magentoTierPriceValue) {
                    $builder->where('product_pricing.price', '!=', DB::raw($magentoTierPriceValue));
                    $builder->orWhereNull('product_pricing.price');
                })
                ->select([
                    'magento_customer_groups.product_pricing_tier_id',
                    'product_listings.product_id',
                    DB::raw($magentoTierPriceValue),
                ]);

            $insertQuery = "INSERT INTO `product_pricing` (`product_pricing_tier_id`, `product_id`, `price`) {$needUpdateQuery->toSql()} ON DUPLICATE KEY UPDATE `price`=VALUES(`price`)";
            DB::insert($insertQuery, $needUpdateQuery->getBindings());
        } catch (QueryException $queryException) {
            // SQLSTATE[40001]: Serialization failure: 1213 Deadlock found when trying to get lock
            $sqlState = $queryException->errorInfo[0] ?? 0;
            // the exception is not Deadlock or this attempt is the last attempt
            if ($sqlState != 40001 || $this->attempts() == $this->tries) {
                $this->fail($queryException);
            } else {
                Helpers::logMysqlDeadlockException($queryException);
                // release again
                $this->release($this->backoff());
            }
        } catch (Throwable $exception) {
            // mark job as failed without releasing it again
            $this->fail($exception);
        }
    }

    /**
     * The unique ID of the job.
     */
    public function uniqueId(): string
    {
        return $this->integrationInstance->id;
    }

    /**
     * Get the cache driver for the unique job lock.
     */
    public function uniqueVia(): Repository
    {
        return Cache::driver('redis');
    }

    /**
     * Calculate the number of seconds to wait before retrying the job.
     */
    public function backoff(): int
    {
        return $this->attempts() * 30;
    }
}
