<?php

namespace App\Jobs\Magento;

use App\Integrations\Magento;
use App\Models\IntegrationInstance;
use App\Models\ProductListing;
use App\Notifications\MonitoringMessage;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Cache\Repository;
use Illuminate\Contracts\Queue\ShouldBeUnique;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Query\JoinClause;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Notification;

class SyncMagentoPricingJob implements ShouldBeUnique, ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public int $uniqueFor = 60 * 60;

    /**
     * Create a new job instance.
     *
     * @return void
     */
    public function __construct(protected IntegrationInstance $integrationInstance)
    {
        //
    }

    /**
     * Sync pricing from SKU to Magento
     * sku.io is the master of price
     */
    public function handle(): void
    {
        $masterOfPrice = strtolower($this->integrationInstance->integration_settings['pricing']['masterOfPrice']['name']);

        // update the price depends on the pricing tier of the integration instance setting
        // get products that need sync
        $query = ProductListing::query()->where('sales_channel_id', $this->integrationInstance->salesChannel->id);
        // master of price is sku.io
        if ($masterOfPrice == ProductListing::MASTER_SKU) {
            $query->where(function (Builder $builder) {
                $builder->whereNull('master_of_price');
                $builder->orWhere('master_of_price', ProductListing::MASTER_SKU);
            });
        } else {
            $query->where('master_of_price', ProductListing::MASTER_SKU);
        }

        // the price is different
        $query->clone()->join('magento_products', function (JoinClause $join) {
            $join->on('magento_products.product', '=', 'product_listings.id')
                ->whereRaw('CAST(`magento_products`.`price` AS FLOAT) != CAST(`product_listings`.`price` AS FLOAT)');
        })->select(['product_listings.id', 'product_listings.price', 'magento_products.sku'])
            ->chunk(20, function (Collection $chunk) {
                // submit price changes to Magento
                $prices = [];
                foreach ($chunk as $listing) {
                    // TODO: store_id always 0?
                    $prices[] = ['price' => $listing->price, 'sku' => $listing->sku, 'store_id' => 0];
                }

                // send request on production mode
                if (config('app.env') !== 'production') {
                    //                return false;
                }

                $response = (new Magento($this->integrationInstance))->updateProductBasePrices($prices);
                if (! empty($response)) {
                    Notification::route('slack', config('slack.debugging'))->notify(new MonitoringMessage('Sync Magento Pricing: '.json_encode($response)));
                }
            });

        // update/add pricing tiers
        $magentoTierPriceValue = "JSON_EXTRACT(`json_object`, JSON_UNQUOTE(REPLACE(JSON_SEARCH(`json_object`, 'one', `magento_customer_groups`.`customer_group_id`, NULL, '$.tier_prices[*].customer_group_id'), 'customer_group_id', 'value')))";
        $magentoTierPriceQty = "JSON_EXTRACT(`json_object`, JSON_UNQUOTE(REPLACE(JSON_SEARCH(`json_object`, 'one', `magento_customer_groups`.`customer_group_id`, NULL, '$.tier_prices[*].customer_group_id'), 'customer_group_id', 'qty')))";
        $query->join('magento_products', 'magento_products.product', '=', 'product_listings.id')
            ->join('product_pricing', 'product_pricing.product_id', '=', 'product_listings.product_id')
            ->join('magento_customer_groups', 'magento_customer_groups.product_pricing_tier_id', 'product_pricing.product_pricing_tier_id')
            ->where(function (Builder $builder) use ($magentoTierPriceValue) {
                $builder->whereRaw("CAST(`product_pricing`.`price` AS FLOAT) != CAST($magentoTierPriceValue AS FLOAT)");
                $builder->orWhereRaw("$magentoTierPriceValue IS NULL");
            })->where(function (Builder $builder) use ($magentoTierPriceQty) {
                // we only check tiers that have qty=1
                $builder->whereRaw("$magentoTierPriceQty = 1");
                $builder->orWhereRaw("$magentoTierPriceQty IS NULL");
            })->select([
                'product_listings.id',
                'magento_products.sku',
                'magento_customer_groups.code',
                'magento_customer_groups.customer_group_id',
                'product_pricing.product_pricing_tier_id',
                'product_pricing.price',
                DB::raw("$magentoTierPriceValue magento_tier_price"),
                DB::raw("JSON_EXTRACT(`json_object`, '$.tier_prices') magento_tier_prices"),
            ])->chunk(20, function (Collection $chunk) {
                $prices = [];
                foreach ($chunk as $index => $listing) {
                    $prices[] = [
                        'customer_group' => $listing->code,
                        'price' => $listing->price,
                        'price_type' => 'fixed', // TODO: always fixed?
                        'quantity' => 1, // TODO: always 1?
                        'sku' => $listing->sku,
                        'website_id' => 0, // TODO: always 0?
                    ];
                }

                // send request on production mode
                if (config('app.env') !== 'production') {
                    //                      return false;
                }

                $response = (new Magento($this->integrationInstance))->updateProductTierPrices($prices);
                if (! empty($response)) {
                    Notification::route('slack', config('slack.debugging'))->notify(new MonitoringMessage('Sync Magento Pricing: '.json_encode($response)));
                }
            });

        /** TODO: delete tiers from magento if the product has tiers that don't exist in sku product
         *      I think we should delete the tier from the product once the user delete it from the sku view(in the same request)
         */
    }

    /**
     * The unique ID of the job.
     */
    public function uniqueId(): string
    {
        return $this->integrationInstance->id;
    }

    /**
     * Get the cache driver for the unique job lock.
     */
    public function uniqueVia(): Repository
    {
        return Cache::driver('redis');
    }
}
