<?php

namespace App\Jobs\Magento;

use App\Jobs\SalesOrderLineMapper;
use App\Jobs\ShipStation\AutoFulfillmentOrder;
use App\Models\Magento\Order;
use App\Models\SalesOrder;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;

/**
 * Class MapSalesOrderLines.
 */
class MapSalesOrderLines extends SalesOrderLineMapper
{
    /**
     * @return Order[]|Builder[]|Collection|mixed
     */
    protected function getOrders()
    {
        $query = Order::query()
            ->where('integration_instance_id', $this->integrationInstance->id)
            ->whereHas('orderLineItems', function (Builder $query) {
                // magento order line has the same listing
                $query->where('product_id', $this->productListing->sales_channel_listing_id);
                // and this magento order line has a sku sales order line whose product_id is null
                $query->whereHas('salesOrderLine', function (Builder $builder) {
                    /** @see SKU-4892 we specified that mappings should not affect historical orders that were already mapped (only those needing mapping). */
                    $builder->whereNull('product_id');
                    // to be sure it refers to the correct sales order line
                    $builder->whereColumn('sales_order_lines.sales_order_id', 'magento_orders.sales_order_id');
                });
            });

        return $query->get();
    }

    public static function handleAfterMapLines(SalesOrder $salesOrder, Order $salesChannelOrder, ?\Illuminate\Support\Collection $skuOrderLines, bool $loadLines = true)
    {
        $loadLines && $salesOrder->load('salesOrderLines');
        // use salesOrderLines because we stop unmap sales order lines
        $productLinesMapped = $salesOrder->salesOrderLines->where('is_product', true)
            ->whereNull('product_id')->isEmpty();

        if ($productLinesMapped) {
            $salesOrder->approve();
            if (! $salesChannelOrder->isPreAuditTrail()) {
                $shipmentsMapped = $salesChannelOrder->createFulfillments($salesOrder);
                $salesChannelOrder->handleReturns($salesOrder);
            }
        }

        // is not PreAuditTrail and the Magento shipments mapped to sku fulfillments
        if (isset($shipmentsMapped) && $shipmentsMapped) {
            // Auto fulfill the sales order if the warehouses of lines belong to the automated warehouses
            (new AutoFulfillmentOrder($salesOrder))->fulfillOpenOrder();
        }
    }
}
