<?php

namespace App\Jobs\Magento;

use App\Enums\DownloadedBy;
use App\Integrations\Magento;
use App\Jobs\Middleware\CheckUniqueJobsMiddleware;
use App\Models\IntegrationInstance;
use App\Models\Magento\Product;
use App\Models\Magento\Product as MagentoProduct;
use App\SDKs\Magento\MagentoException;
use Carbon\Carbon;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Cache\Repository;
use Illuminate\Contracts\Queue\ShouldBeUnique;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Str;

class GetProducts implements ShouldBeUnique, ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * The number of seconds after which the job's unique lock will be released.
     *
     * @var int
     */
    public $uniqueFor = 3600;

    /**
     * @var array[]
     */
    protected $summary = [
        'New Products' => 0,
        'Total Products' => 0,
    ];

    /**
     * Create a new job instance.
     *
     * @return void
     */
    public function __construct(protected IntegrationInstance $integrationInstance, protected array $options = [])
    {
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        set_time_limit(0);
        if (function_exists('pcntl_async_signals')) {
        }

        $magento = new Magento($this->integrationInstance);

        try {
            foreach ($magento->getProducts($this->getMagentoSearchCriteria()) as [$response, $nextSearchCriteria]) {
                $this->options = $nextSearchCriteria;

                foreach ($response['items'] as $magento_product) {
                    $this->createByArray($magento_product);

                    $this->summary['Total Products'] += 1;
                }
            }
        } catch (MagentoException $magentoException) {
            // it will dispatch again if the exception is "cURL Error" otherwise it will fail immediately
            if (Str::startsWith($magentoException->getMessage(), 'cURL Error')) {
                // delete the unique key
                $this->uniqueVia()->lock(
                    'laravel_unique_job:'.get_class($this).$this->uniqueId()
                )->forceRelease();
                // dispatch again
                dispatch(new static($this->integrationInstance, $this->options))->onQueue($this->queue)->delay(120);

                return;
            }
            throw $magentoException;
        }

        // auto create products if the setting is enabled
        if (($this->integrationInstance->integration_settings['auto_create_products'] ?? false) === true) {
            dispatch(new CreateSkuProducts($this->integrationInstance))->onQueue($this->queue);
        }

        // run syncing pricing after products is fetched to ensure that magento prices is updated
        // Sync pricing from Magento To SKU
        dispatch(new SyncSkuPricingJob($this->integrationInstance))->onQueue($this->queue);
        // Sync pricing from SKU to Magento
        dispatch(new SyncMagentoPricingJob($this->integrationInstance))->onQueue($this->queue);

        return $this->summary;
    }

    private function createByArray(array $magentoProduct)
    {
        // we only add the simple(simple, virtual and downloadable) products
        // for the bundle product, the user should create it manually on the sku them map it
        if (in_array($magentoProduct['type_id'], ['configurable', 'grouped'])) {
            return;
        }

        // to ignore the duplication on unique keys (variant_id/sku with integration_instance_id)
        MagentoProduct::query()->insertOrIgnore([
            'variant_id' => $magentoProduct['id'],
            'integration_instance_id' => $this->integrationInstance->id,
            'json_object' => json_encode($magentoProduct),
            'updated_by' => $this->options['downloaded_by'] ?? DownloadedBy::Job,
        ]);

        // get the magento product to update _id and product listing
        $product = MagentoProduct::query()->firstWhere([
            'variant_id' => $magentoProduct['id'],
            'integration_instance_id' => $this->integrationInstance->id,
        ]);

        // if the product is ignored
        if ($product) {
            $product->_id = $product->id;
            $product->update();

            $this->updateListingMapping($product);
        }
    }

    private function updateListingMapping(MagentoProduct $product)
    {
        $product->productListing()->update(['document_id' => $product->id, 'listing_sku' => $product->sku, 'title' => $product->name]);
    }

    /**
     * Parse job/integrationInstance options to Magento searchCriteria.
     *
     * @see https://devdocs.magento.com/guides/v2.4/rest/performing-searches.html
     */
    private function getMagentoSearchCriteria(): array
    {
        $options = Arr::only($this->options, ['filterGroups', 'pageSize']);
        // if the searchCriteria already sent and parsed
        if (! empty($options)) {
            return $options;
        }

        // custom options
        if (! empty(Arr::only(array_filter($this->options), ['createdAfter', 'createdBefore', 'lastUpdatedAfter', 'lastUpdatedBefore']))) {
            if (isset($this->options['createdAfter'])) {
                $options['filterGroups'][]['filters'][0] = [
                    'field' => 'created_at',
                    'conditionType' => 'gt',
                    'value' => Carbon::parse($this->options['createdAfter'])->toIso8601String(),
                ];
            }
            if (isset($this->options['createdBefore'])) {
                $options['filterGroups'][]['filters'][0] = [
                    'field' => 'created_at',
                    'conditionType' => 'lt',
                    'value' => Carbon::parse($this->options['createdBefore'])->toIso8601String(),
                ];
            }
            if (isset($this->options['lastUpdatedAfter'])) {
                $options['filterGroups'][]['filters'][0] = [
                    'field' => 'updated_at',
                    'conditionType' => 'gt',
                    'value' => Carbon::parse($this->options['lastUpdatedAfter'])->toIso8601String(),
                ];
            }
            if (isset($this->options['lastUpdatedBefore'])) {
                $options['filterGroups'][]['filters'][0] = [
                    'field' => 'updated_at',
                    'conditionType' => 'lt',
                    'value' => Carbon::parse($this->options['lastUpdatedBefore'])->toIso8601String(),
                ];
            }
        } elseif (isset($this->options['lastUpdatedAfterLastFetched']) && $this->options['lastUpdatedAfterLastFetched']) {
            if ($latestProduct = Product::latestProduct($this->integrationInstance->id, DownloadedBy::Job)->first()) {
                $field = 'updated_at';
                $value = Carbon::parse($latestProduct->json_object['updated_at'])->addSecond();

                $options['filterGroups'][]['filters'][0] = [
                    'field' => $field,
                    'conditionType' => 'gteq', // Greater than or equal
                    'value' => $value->toIso8601String(),
                ];
            }
        }

        // limit per page
        $options['pageSize'] = (int) ($this->options['limit'] ?? 300);

        return $options;
    }

    /**
     * The unique ID of the job.
     */
    public function uniqueId(): string
    {
        return $this->integrationInstance->id;
    }

    /**
     * Get the cache driver for the unique job lock.
     */
    public function uniqueVia(): Repository
    {
        return Cache::driver('redis');
    }

    /**
     * Get the middleware the job should pass through.
     */
    public function middleware(): array
    {
        return [
            new CheckUniqueJobsMiddleware([
                new SyncSkuPricingJob($this->integrationInstance),
                new SyncMagentoPricingJob($this->integrationInstance),
            ]),
        ];
    }
}
