<?php

namespace App\Jobs\Magento;

use App\Enums\DownloadedBy;
use App\Integrations\Magento;
use App\Models\IntegrationInstance;
use App\Models\Magento\Order;
use App\Models\Magento\OrderLineItem;
use Carbon\Carbon;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Arr;
use Throwable;

class GetOrdersJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * Create a new job instance.
     *
     * @return void
     */
    public function __construct(public IntegrationInstance $integrationInstance, public array $options = [])
    {
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        $magento = new Magento($this->integrationInstance);

        foreach ($magento->getSalesOrders($this->getMagentoSearchCriteria()) as $response) {
            $orderIds = collect($response['items'])->pluck('entity_id')->toArray();
            // get CreditMemos, Shipments and Returns
            $creditMemos = collect($this->getCreditMemos($orderIds));
            $shipments = collect($this->getShipments($orderIds));
            $returns = collect($this->getReturns($orderIds));
            $invoices = collect($this->getInvoices($orderIds));

            foreach ($response['items'] as $order) {
                // add creditMemos, Shipment, Returns and Invoices to the order
                $order['credit_memos'] = $creditMemos->where('order_id', $order['entity_id'])->values()->all();
                $order['shipments'] = $shipments->where('order_id', $order['entity_id'])->values()->all();
                $order['returns'] = $returns->where('order_id', $order['entity_id'])->values()->all();
                $order['invoices'] = $invoices->where('order_id', $order['entity_id'])->values()->all();
                // parse to sku Sales Order
                $this->handleOrder($order);
            }
        }
    }

    /**
     * Magento order to sku sales order
     */
    private function handleOrder(array $order): void
    {
        if ($magentoOrder = $this->getMagentoOrder($order)) {
            $lastUpdated = Carbon::parse($magentoOrder->json_object['updated_at'], 'UTC');
        }

        Order::query()->upsert([
            'integration_instance_id' => $this->integrationInstance->id,
            'json_object' => json_encode($order),
            'downloaded_by' => $this->options['downloaded_by'] ?? DownloadedBy::Job,
            'updated_by' => $this->options['downloaded_by'] ?? DownloadedBy::Job,
        ], [], ['updated_by', 'json_object']);

        // get again after saving
        $magentoOrder = $this->getMagentoOrder($order);
        // add line items
        OrderLineItem::query()->upsert(array_map(fn ($line) => ['magento_order_id' => $magentoOrder->id, 'line_item' => json_encode($line)], $magentoOrder->json_object['items']), []);

        if ($magentoOrder->isOpen() && (! $this->integrationInstance->open_start_date || $this->integrationInstance->open_start_date > Carbon::parse($order['created_at'], 'UTC'))) {
            return;
        }

        if (! $magentoOrder->isOpen() && (! $this->integrationInstance->closed_start_date || $this->integrationInstance->closed_start_date > Carbon::parse($order['created_at'], 'UTC'))) {
            return;
        }

        /** @see SKU-4130 exclude archived orders when running process orders */
        if ($magentoOrder->isArchived()) {
            return;
        }

        try {
            $magentoOrder->setRelation('integrationInstance', $this->integrationInstance);

            if ($magentoOrder->wasRecentlyCreated || ! $magentoOrder->salesOrder) {
                $magentoOrder->createSKUOrder();
            } elseif (Carbon::parse($order['updated_at'], 'UTC')->greaterThan($lastUpdated)) {
                $magentoOrder->updateSKUOrder();
            }
        } catch (Throwable $e) {
            $magentoOrder->errors = [
                'message' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'code' => $e->getCode(),
            ];
            $magentoOrder->save();
        }
    }

    private function getMagentoOrder(array $order): ?Order
    {
        return Order::query()->firstWhere([
            'integration_instance_id' => $this->integrationInstance->id,
            'increment_id' => $order['increment_id'],
        ]);
    }

    /**
     * Get Credit Memos by order ids
     */
    private function getCreditMemos(array $orderIds): array
    {
        $options['pageSize'] = 300; // maximum
        $options['filterGroups'][]['filters'][0] = [
            'field' => 'order_id',
            'conditionType' => 'in',
            'value' => implode(',', $orderIds),
        ];

        $magento = new Magento($this->integrationInstance);
        $creditMemos = [];
        foreach ($magento->getCreditMemos($options) as $response) {
            $creditMemos = array_merge($creditMemos, $response['items']);
        }

        return $creditMemos;
    }

    /**
     * Get Shipments by order ids
     */
    private function getShipments(array $orderIds): array
    {
        $options['pageSize'] = 300; // maximum
        $options['filterGroups'][]['filters'][0] = [
            'field' => 'order_id',
            'conditionType' => 'in',
            'value' => implode(',', $orderIds),
        ];

        $magento = new Magento($this->integrationInstance);
        $shipments = [];
        foreach ($magento->getShipments($options) as $response) {
            $shipments = array_merge($shipments, $response['items']);
        }

        return $shipments;
    }

    /**
     * Get Returns(RMA) by order ids
     */
    private function getReturns(array $orderIds): array
    {
        $options['pageSize'] = 300; // maximum
        $options['filterGroups'][]['filters'][0] = [
            'field' => 'order_id',
            'conditionType' => 'in',
            'value' => implode(',', $orderIds),
        ];

        $magento = new Magento($this->integrationInstance);
        $returns = [];
        foreach ($magento->getReturns($options) as $response) {
            $returns = array_merge($returns, $response['items']);
        }

        return $returns;
    }

    /**
     * Get Invoices by order ids
     */
    private function getInvoices(array $orderIds): array
    {
        $options['pageSize'] = 300; // maximum
        $options['filterGroups'][]['filters'][0] = [
            'field' => 'order_id',
            'conditionType' => 'in',
            'value' => implode(',', $orderIds),
        ];

        $magento = new Magento($this->integrationInstance);
        $invoices = [];
        foreach ($magento->getInvoices($options) as $response) {
            $invoices = array_merge($invoices, $response['items']);
        }

        return $invoices;
    }

    /**
     * Parse job/integrationInstance options to Magento searchCriteria.
     *
     * @see https://devdocs.magento.com/guides/v2.4/rest/performing-searches.html
     */
    private function getMagentoSearchCriteria(): array
    {
        // get by order increment ids
        if (! empty($this->options['ids'])) {
            $options['filterGroups'][]['filters'][0] = [
                'field' => 'increment_id',
                'conditionType' => 'in',
                'value' => implode(',', $this->options['ids']),
            ];

            return $options;
        }

        // custom options
        if (! empty(Arr::only(array_filter($this->options), ['createdAfter', 'createdBefore', 'lastUpdatedAfter', 'lastUpdatedBefore']))) {
            if (isset($this->options['createdAfter'])) {
                $options['filterGroups'][]['filters'][0] = [
                    'field' => 'created_at',
                    'conditionType' => 'gt',
                    'value' => Carbon::parse($this->options['createdAfter'])->toIso8601String(),
                ];
            }
            if (isset($this->options['createdBefore'])) {
                $options['filterGroups'][]['filters'][0] = [
                    'field' => 'created_at',
                    'conditionType' => 'lt',
                    'value' => Carbon::parse($this->options['createdBefore'])->toIso8601String(),
                ];
            }
            if (isset($this->options['lastUpdatedAfter'])) {
                $options['filterGroups'][]['filters'][0] = [
                    'field' => 'updated_at',
                    'conditionType' => 'gt',
                    'value' => Carbon::parse($this->options['lastUpdatedAfter'])->toIso8601String(),
                ];
            }
            if (isset($this->options['lastUpdatedBefore'])) {
                $options['filterGroups'][]['filters'][0] = [
                    'field' => 'updated_at',
                    'conditionType' => 'lt',
                    'value' => Carbon::parse($this->options['lastUpdatedBefore'])->toIso8601String(),
                ];
            }
        }
        // integration instance options
        else {
            $field = 'created_at';
            $value = min($this->integrationInstance->open_start_date, $this->integrationInstance->closed_start_date);

            if ($latestOrder = Order::latestOrder($this->integrationInstance->id, DownloadedBy::Job, 'json_object->updated_at')->first()) {
                $field = 'updated_at';
                $value = Carbon::parse($latestOrder->json_object['updated_at'])->addSecond();
            }

            $options['filterGroups'][]['filters'][0] = [
                'field' => $field,
                'conditionType' => 'gteq', // Greater than or equal
                'value' => $value->toIso8601String(),
            ];
        }

        // custom status
        if (! empty($this->options['status'])) {
            $options['filterGroups'][]['filters'][0] = [
                'field' => 'status',
                'conditionType' => 'in',
                'value' => implode(',', $this->options['status']),
            ];
        }
        // limit per page
        $options['pageSize'] = (int) ($this->options['limit'] ?? 300);

        return $options;
    }
}
