<?php

namespace App\Jobs;

use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use PHPJasper\Exception\ErrorCommandExecutable;
use PHPJasper\Exception\InvalidCommandExecutable;
use PHPJasper\Exception\InvalidFormat;
use PHPJasper\Exception\InvalidInputFile;
use PHPJasper\Exception\InvalidResourceDirectory;
use PHPJasper\PHPJasper;

trait JasperExport
{
    abstract protected function defaultJasperTemplateFile(): string;

    /**
     * Get Jasper template file path.
     */
    private function getTemplatePath(?string $templateFile): string
    {
        if (empty($templateFile)) {
            return resource_path("views/jasper_files/{$this->defaultJasperTemplateFile()}");
        }

        return storage_path(config('uploader.jrxml.target').$templateFile);
    }

    /**
     * Export PDF file from Jasper file.
     *
     *
     * @throws ErrorCommandExecutable
     * @throws InvalidCommandExecutable
     * @throws InvalidFormat
     * @throws InvalidInputFile
     * @throws InvalidResourceDirectory
     */
    public function export(array $items, string $templateFile, string $outputFile): string
    {
        $dataFile = time().rand(1000, 9999).'.json';
        // save items to json file
        Storage::disk('jasper_reports')->put($dataFile, json_encode(['items' => $items]));

        // export options
        $options = [
            'format' => ['pdf'],
            'locale' => 'en',
            'db_connection' => [
                'driver' => 'json',
                'data_file' => Storage::disk('jasper_reports')->path($dataFile),
                'json_query' => 'items',
            ],
        ];

        $outputFile = Str::endsWith($outputFile, '.pdf') ? Str::replaceLast('.pdf', '', $outputFile) : $outputFile;

        try {
            // export
            (new PHPJasper())->process($templateFile, $outputFile, $options)->execute();
            @chmod("{$outputFile}.pdf", 0666);
        } finally {
            Storage::disk('jasper_reports')->delete($dataFile);
        }

        return $outputFile.'.pdf';
    }
}
