<?php

namespace App\Jobs;

use App\Models\Product;
use App\Models\Supplier;
use App\Models\SupplierInventory;
use App\Models\SupplierProduct;
use Illuminate\Bus\Batchable;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use InvalidArgumentException;

class ImportSupplierInventory implements ShouldQueue
{
    use Batchable, Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    private array $supplierSkus = [];

    public function __construct(public array $rows, public array $meta = [])
    {
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        // Set source to user
        $this->rows = array_map(function ($row) {
            $row['source'] = SupplierInventory::SOURCE_USER;
            $row['supplier_id'] = $this->getSupplierId();
            foreach (array_keys($row) as $key) {
                if ($row[$key] === 'NULL') {
                    $row[$key] = null;
                }
            }
            if (! empty($row['quantity'])) {
                $row['quantity'] = (int) ($row['quantity']);
            }

            return $row;
        }, $this->rows);

        // We update for all those with ids in a bulk quantity.
        $withIds = array_filter($this->rows, fn ($row) => ! empty($row['id']));

        if (! empty($withIds)) {
            $this->updateWithIds($withIds);
        }

        // Attempt to match by sku
        $withoutIdsButWithSku = array_filter(
            array_udiff($this->rows, $withIds, fn ($a, $b) => $a <=> $b),
            fn ($row) => ! empty($row['sku']) && empty($row['id'])
        );

        if (! empty($withoutIdsButWithSku)) {
            $this->importWithSku($withoutIdsButWithSku);
        }

        // Create new records for remaining rows
        $remaining = array_udiff($this->rows, $withIds, $withoutIdsButWithSku, fn ($a, $b) => $a <=> $b);
        if (! empty($remaining)) {
            $this->importFreshRecords($remaining);
        }

        // Update supplier skus
        if (! empty($this->supplierSkus)) {
            $this->updateSupplierSkus();
        }
    }

    public function getSupplierId(): int
    {
        if (empty($this->meta['supplier_id'])) {
            throw new InvalidArgumentException('Supplier Inventory import missing supplier_id');
        }

        return $this->meta['supplier_id'];
    }

    private function importWithSku(array $withoutIdsButWithSku): void
    {
        // Fill in product ids for those with only sku
        $withoutProductIds = array_filter($withoutIdsButWithSku, fn ($row) => empty($row['product_id']));
        $products = Product::query()
            ->whereIn('sku', collect($withoutProductIds)->pluck('sku')->toArray())
            ->get(['id', 'sku'])
            ->toArray();

        $mapping = array_combine(array_column($products, 'sku'), array_column($products, 'id'));
        $results = collect($withoutIdsButWithSku)->map(function ($row) use ($mapping) {
            if (! empty($mapping[$row['sku']])) {
                $row['product_id'] = $mapping[$row['sku']];
                $row['supplier_id'] = $this->getSupplierId();
                unset($row['warehouse_id'], $row['sku']);

                return $row;
            }

            return false;
        })->filter()->merge(collect($withoutIdsButWithSku)->filter(fn ($row) => ! empty($row['product_id'])))
            ->toArray();

        // Update
        batch()->updateWithTwoIndex(
            new SupplierInventory,
            $this->makeUpdateData($results),
            'product_id',
            'supplier_id'
        );
    }

    private function updateWithIds(array $withIds): void
    {
        // If we have skus and supplier_skus, we bind in
        // product ids so we can update supplier_sku.
        $headers = array_keys($withIds[0]);
        if (in_array('supplier_sku', $headers)) {
            $inventories = SupplierInventory::query()
                ->whereIn('id', collect($withIds)->pluck('id')->toArray())
                ->get(['id', 'product_id'])
                ->toArray();
            $mapping = array_combine(array_column($inventories, 'id'), array_column($inventories, 'product_id'));
            $results = collect($withIds)->map(function ($row) use ($mapping) {
                if (! empty($mapping[$row['id']])) {
                    $row['product_id'] = $mapping[$row['id']];
                    // Remove fields that cannot be updated
                    unset($row['supplier_id'], $row['warehouse_id']);

                    return $row;
                }

                return false;
            })->filter()->toArray();
        } else {
            $results = array_map(function ($row) {
                // Remove fields that cannot be updated
                unset($row['supplier_id'], $row['warehouse_id']);

                return $row;
            }, $withIds);
        }

        batch()->update(
            new SupplierInventory,
            $this->makeUpdateData($results),
            'id'
        );
    }

    private function importFreshRecords(array $rows): void
    {
        // Ensure that there are product ids
        $rows = array_filter($rows, function ($row) {
            return ! empty($row['product_id']);
        });

        if (empty($rows)) {
            return;
        }

        /** @var Supplier $supplier */
        $supplier = Supplier::query()->findOrFail($this->getSupplierId());

        SupplierInventory::query()->upsert(
            array_map(function ($row) use ($supplier) {
                $this->trackSupplierSku($row);
                if (empty($row['warehouse_id'])) {
                    $row['warehouse_id'] = $supplier->default_warehouse_id;
                }
                unset($row['sku'], $row['supplier_sku']);

                return $row;
            }, $rows),
            ['product_id', 'supplier_id', 'warehouse_id']
        );
    }

    private function makeUpdateData(array $rows): array
    {
        $keys = array_merge((new SupplierInventory)->getFillable(), ['id']);

        return array_map(function ($row) use ($keys) {
            $this->trackSupplierSku($row);

            return array_intersect_key(
                $row,
                array_flip(array_filter($keys, fn ($key) => in_array($key, array_keys($row))))
            );
        }, $rows);
    }

    private function trackSupplierSku(array $row): void
    {
        if (! empty($row['supplier_sku']) && ! empty($row['product_id'])) {
            $this->supplierSkus[] = [
                'product_id' => $row['product_id'],
                'supplier_sku' => trim(str_replace('"', '', preg_replace('/\s+/', ' ', $row['supplier_sku']))),
                'supplier_id' => $this->getSupplierId(),
            ];
        }
    }

    private function updateSupplierSkus(): void
    {
        batch()->updateWithTwoIndex(
            new SupplierProduct,
            $this->supplierSkus,
            'product_id',
            'supplier_id'
        );
    }
}
