<?php

namespace App\Jobs;

use App\Exporters\Jasper\PackingSlipTransformer;
use App\Ghostscript\Ghostscript;
use App\Models\PackingSlipQueue;
use App\Models\SalesOrder;
use App\Response;
use Exception;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Storage;
use InvalidArgumentException;
use PHPJasper\Exception\ErrorCommandExecutable;
use PHPJasper\Exception\InvalidCommandExecutable;
use PHPJasper\Exception\InvalidFormat;
use PHPJasper\Exception\InvalidInputFile;
use PHPJasper\Exception\InvalidResourceDirectory;

class GenerateSalesOrderPackingSlipsJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, JasperExport, Queueable, SerializesModels;

    /**
     * Create a new job instance.
     *
     * @return void
     */
    private $response = [];

    private $errorsCount = 0;

    private $oneFile = false;

    private $markAsPrinted = false;

    public function __construct(protected array|int $salesOrderIds, protected ?PackingSlipQueue $packingSlipQueue = null, protected array $options = [])
    {
        $this->salesOrderIds = Arr::wrap($salesOrderIds);
        $this->oneFile = ! empty($options['one_file']) && count($this->salesOrderIds) > 1;
        $this->markAsPrinted = ! empty($options['mark_as_printed']);

        if (empty($this->salesOrderIds)) {
            throw new InvalidArgumentException('Empty sales order ids');
        }
    }

    /**
     * Execute the job.
     *
     *
     * @throws ErrorCommandExecutable
     * @throws InvalidCommandExecutable
     * @throws InvalidFormat
     * @throws InvalidInputFile
     * @throws InvalidResourceDirectory
     * @throws Exception
     */
    public function handle(): ?array
    {
        $this->packingSlipQueue?->printing();
        $this->response['generatedFile'] = null;
        /** @var Collection|SalesOrder[] $salesOrders */
        $salesOrders = SalesOrder::with([
            'store.address',
            'customer',
            'shippingAddress',
            'billingAddress',
            'shippingMethod',
            'notes',
            'salesOrderLines.product.productAttributes',
            'salesOrderLines.product.primaryImage',
            'salesOrderLines.activeBackorderQueue',
        ])->findMany($this->salesOrderIds);

        $generatedFiles = [];
        $allLineItems = [];
        foreach ($salesOrders->groupBy('active_store.custom_jrxml_file') as $templateFile => $salesOrders) {
            foreach ($salesOrders as $salesOrder) {
                if (is_null($salesOrder->customer)) {
                    $this->response['errors'][$salesOrder->id] = Response::getError('This sales order does not contain customer', Response::CODE_ARE_NOT_PRINTED, 'id');
                    $this->errorsCount++;

                    continue;
                }
                if ($salesOrder->salesOrderLines->where('is_product', true)->isEmpty()) {
                    $this->response['errors'][$salesOrder->id] = Response::getError('This sales order does not contain products', Response::CODE_ARE_NOT_PRINTED, 'id');
                    $this->errorsCount++;

                    continue;
                }

                $orderItems = [];
                $orderData = PackingSlipTransformer::salesOrder($salesOrder);

                foreach ($salesOrder->salesOrderLines->where('is_product', true) as $salesOrderLine) {
                    foreach (PackingSlipTransformer::salesOrderLine($salesOrderLine) as $line) {
                        if ($line) {
                            $orderItems[] = array_merge($line, $orderData);
                        }
                    }
                }

                if ($this->oneFile) {
                    $allLineItems[$templateFile] = array_merge($allLineItems[$templateFile] ?? [], $orderItems);
                } else {
                    $orderPSFile = Storage::disk('order_packing_slips')->path($salesOrder->id);
                    $generatedFiles[] = $this->export($orderItems, $this->getTemplatePath($templateFile), $orderPSFile);
                }
            }
        }

        if ($this->oneFile) {
            foreach ($allLineItems as $templateFile => $items) {
                $orderPSFile = Storage::disk('order_packing_slips')->path(time().rand(1000, 9999));
                $generatedFiles[] = $this->export($items, $this->getTemplatePath($templateFile), $orderPSFile);
            }
        }

        if ($this->markAsPrinted) {
            SalesOrder::whereIn('id', array_diff($this->salesOrderIds, array_keys($this->response['errors'] ?? [])))->update([
                'packing_slip_printed_at' => now(),
            ]);
        }

        // mark packing slip queue as printed (delete it)
        $this->packingSlipQueue?->printed();

        if (empty($generatedFiles)) {
            return $this->response;
        }

        // if only one file, return it
        if (count($generatedFiles) == 1) {
            $this->response['generatedFile'] = $generatedFiles[0];

            return $this->response;
        }

        // if more than one file, combine them in one file and return it
        $outputFile = Storage::disk('order_packing_slips')->path(time().rand(1000, 9999).'.pdf');
        if (is_null($combineStatus = (new Ghostscript($outputFile, $generatedFiles))->combine())) {
            $this->response['generatedFile'] = $outputFile;

            return $this->response;
        }

        throw new Exception("Ghostscript error: {$combineStatus}");
    }

    protected function defaultJasperTemplateFile(): string
    {
        return 'SKU_Packing_Slip.jrxml';
    }

    public function tags(): array
    {
        $tags = ['ids:'.implode(',', $this->salesOrderIds)];
        if ($this->packingSlipQueue) {
            $tags[] = "PackingSlipQueue:{$this->packingSlipQueue->id}";
        }

        return $tags;
    }
}
