<?php

namespace App\Jobs;

use App\Models\InventoryForecast;
use App\Models\Supplier;
use App\Repositories\InventoryForecastRepository;
use App\Services\InventoryForecasting\ForecastManager;
use App\Services\PurchaseOrder\PurchaseOrderBuilder\Exceptions\LargePurchaseOrderException;
use Illuminate\Bus\Batchable;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Throwable;

class ForecastSupplierInventoryJob implements ShouldQueue
{
    use Dispatchable;
    use InteractsWithQueue;
    use Queueable;
    use SerializesModels;
    use Batchable;

    private bool $buildingForSpecificProducts = false;

    /**
     * ForecastSupplierInventory constructor.
     */
    public function __construct(
        private readonly ForecastManager                  $manager,
        private readonly InventoryForecastRepository $cache,
        private readonly Supplier                         $supplier,
        private readonly array                            $productIds = []
    )
    {}

    /**
     * Execute the job.
     */
    public function handle(): int
    {
        try {
            $this->manager->forecastForSupplier(supplier: $this->supplier);
        } catch (LargePurchaseOrderException|BindingResolutionException $e) {
            slack('Forecasting failed for supplier ' . $this->supplier->name . ' with message: ' . $e->getMessage());
        }

        return 0;
    }

    public function failed(Throwable $exception)
    {
        $this->cache->clearForSupplier(
            supplierId: $this->supplier->id,
            productIds: $this->buildingForSpecificProducts ? $this->productIds : []
        );
    }
}
