<?php

namespace App\Jobs\Ebay;

use App\Events\SalesChannelListingLinked;
use App\Helpers;
use App\Integrations\Channel;
use App\Models\Attribute;
use App\Models\Product;
use App\Models\ProductListing;
use App\Models\SalesChannel;
use App\Models\SalesChannelType;
use App\Models\Setting;
use DTS\eBaySDK\Trading\Enums\AckCodeType;
use Exception;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

class GetProducts implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * @var SalesChannel
     */
    protected $salesChannel;

    /**
     * Create a new job instance.
     */
    public function __construct(SalesChannel $salesChannel)
    {
        $this->salesChannel = $salesChannel;
    }

    /**
     * Execute the job.
     *
     * @throws Exception
     */
    public function handle(): void
    {
        /**
         * Get instance of channel type.
         */
        $ebay_channel = Channel::getInstance(SalesChannelType::TYPE_EBAY, $this->salesChannel->credentials);

        /**
         * Fetch products.
         */
        $products_generator = $ebay_channel->getProducts(['Pagination' => ['EntriesPerPage' => 10]]);

        foreach ($products_generator as $products_response) {
            /**
             * Currently just print errors.
             */
            if (isset($products_response->Errors)) {
                foreach ($products_response->Errors as $error) {
                    Log::error($error->ShortMessage.'('.$error->ErrorCode.'): '.$error->LongMessage);
                }
            }

            /**
             * Store products.
             */
            if ($products_response->Ack !== AckCodeType::C_FAILURE) {
                /**
                 * Basic attributes.
                 */
                $description_attribute = Attribute::firstOrCreate(
                    ['name' => 'description'],
                    [
                        'type' => Attribute::TYPE_LONGTEXT,
                        'has_options' => false,
                    ]
                );

                $image_attribute = Attribute::firstOrCreate(
                    ['name' => 'image'],
                    [
                        'type' => Attribute::TYPE_LONGTEXT,
                        'has_options' => false,
                    ]
                );

                $condition_attribute = Attribute::firstOrCreate(
                    ['name' => 'condition'],
                    [
                        'type' => Attribute::TYPE_STRING,
                        'has_options' => false,
                    ]
                );

                $currency_attribute = Attribute::firstOrCreate(
                    ['name' => 'currency'],
                    [
                        'type' => Attribute::TYPE_STRING,
                        'has_options' => false,
                    ]
                );

                foreach ($products_response->ItemArray->Item as $ebay_product) {
                    //          $ebay_product = new ItemType();

                    /**
                     * Get from DB or New object of ProductListing by "sales channel" and "product listing id".
                     */
                    $product_listing = ProductListing::firstOrNew([
                        'sales_channel_id' => $this->salesChannel->id,
                        'sales_channel_listing_id' => $ebay_product->ItemID,
                    ]);
                    /**
                     * Check if product listing exists on DB.
                     * Get Product by id (product listing exists), by sku (product listing not exists) or New object.
                     */
                    if ($product_listing->exists) {
                        $product = Product::findOrNew($product_listing->product_id);
                    } else {
                        $product = Product::whereNotNull('sku')->where('sku', $ebay_product->SKU ?? null)->firstOrNew([]);

                        if ($product->exists) {
                            event(new SalesChannelListingLinked($product, $this->salesChannel));
                        }
                    }

                    /**
                     * Fill Product attributes and save on DB.
                     */
                    $defaultWeightUnit = Helpers::setting(Setting::KEY_DEFAULT_WEIGHT_UNIT, null, true);
                    $defaultDimensionUnit = Helpers::setting(Setting::KEY_DEFAULT_DIMENSION_UNIT, null, true);

                    $product->type = Product::TYPE_STANDARD;
                    $product->sku = $ebay_product->SKU;
                    $product->name = $ebay_product->Title;
                    $product->weight = $ebay_product->ShippingPackageDetails->WeightMajor->value ?? 0;
                    $product->weight_unit = $ebay_product->ShippingPackageDetails->WeightMajor->unit ?? $defaultWeightUnit;
                    $product->length = $ebay_product->ShippingPackageDetails->PackageLength->value ?? 0;
                    $product->width = $ebay_product->ShippingPackageDetails->PackageWidth->value ?? 0;
                    $product->height = $ebay_product->ShippingPackageDetails->PackageDepth->value ?? 0;
                    $product->dimension_unit = $ebay_product->ShippingPackageDetails->PackageLength->unit ?? $defaultDimensionUnit;

                    $product->case_dimension_unit = $defaultDimensionUnit;
                    $product->case_weight_unit = $defaultWeightUnit;
                    $product->save();

                    /**
                     * Fill Product Listing attributes and save on DB.
                     */
                    $product_listing->product_id = $product->id;
                    $product_listing->listing_sku = $ebay_product->SKU;
                    $product_listing->price = $ebay_product->StartPrice->value;
                    $product_listing->quantity = $ebay_product->Quantity - $ebay_product->SellingStatus->QuantitySold;
                    $product_listing->save();

                    /**
                     * Sync Product attributes and values.
                     */
                    $product->productAttributes()->syncWithoutDetaching(
                        [
                            $description_attribute->id => ['value_longtext' => $ebay_product->Description],
                            $image_attribute->id => ['value_longtext' => $ebay_product->PictureDetails->GalleryURL],
                            $condition_attribute->id => ['value_string' => $ebay_product->ConditionDisplayName],
                            $currency_attribute->id => ['value_string' => $ebay_product->StartPrice->currencyID],
                        ]
                    );

                    /**
                     * Store Product variations by same way of parent product.
                     */
                    foreach ($ebay_product->Variations->Variation ?? [] as $ebay_variation) {
                        /**
                         * Variation listing id from EAN or ISBN or UPC.
                         */
                        $variationID = $ebay_variation->VariationProductListingDetails->EAN ??
                           $ebay_variation->VariationProductListingDetails->ISBN ??
                           $ebay_variation->VariationProductListingDetails->UPC;

                        /**
                         * Currently just Logging without save variation on DB.
                         */
                        if (empty($variationID) && empty($ebay_variation->SKU)) {
                            Log::error('Product ('.$product->id.') have variations without ID (sku, ean, isbn and upc)');

                            continue;
                        }

                        $variation_product_listing = ProductListing::firstOrNew([
                            'sales_channel_id' => $this->salesChannel->id,
                            'sales_channel_listing_id' => $variationID,
                        ]);

                        if ($variation_product_listing->exists) {
                            $product_variation = Product::find($variation_product_listing->product_id) ?? $product->replicate();
                        } else {
                            $product_variation = Product::whereNotNull('sku')->where('sku', $ebay_variation->SKU ?? null)->first() ?? $product->replicate();
                        }

                        $product_variation->parent_id = $product->id;
                        $product_variation->sku = $ebay_variation->SKU ?? null;
                        $product_variation->gtin = [
                            'upc' => $ebay_variation->VariationProductListingDetails->UPC ?? null,
                            'ean' => $ebay_variation->VariationProductListingDetails->EAN ?? null,
                            'isbn' => $ebay_variation->VariationProductListingDetails->ISBN ?? null,
                        ];
                        $product_variation->save();

                        $variation_product_listing->product_id = $product_variation->id;
                        $variation_product_listing->listing_sku = $ebay_variation->SKU ?? null;
                        $variation_product_listing->price = $ebay_variation->StartPrice->value;
                        $variation_product_listing->quantity = floatval($ebay_product->Quantity - $ebay_product->SellingStatus->QuantitySold);
                        $variation_product_listing->save();

                        /**
                         * Store product/variation attributes and values.
                         */
                        foreach ($ebay_variation->VariationSpecifics as $specifics) {
                            $variation_attribute = Attribute::firstOrCreate(
                                ['name' => $specifics->NameValueList[0]->Name],
                                [
                                    'type' => Attribute::TYPE_STRING,
                                    'has_options' => true,
                                ]
                            );

                            $variation_attribute->values()->firstOrCreate(['value' => $specifics->NameValueList[0]->Value[0]]);

                            $product_variation->productAttributes()->syncWithoutDetaching([$variation_attribute->id => ['value_string' => $specifics->NameValueList[0]->Value[0]]]);
                        }
                    }

                    \App\Models\Ebay\Product::updateOrCreate(['ItemID' => $ebay_product->ItemID], $ebay_product->toArray());
                }
            }
        }
    }
}
