<?php

namespace App\Jobs\DataFeed;

use App\Facades\ImportFile;
use App\Models\DataFeed;
use App\Models\Product;
use App\Models\ProductPricing;
use App\Models\ProductPricingTier;
use App\Models\Supplier;
use App\Models\SupplierPricingTier;
use App\Models\SupplierProduct;
use App\Models\SupplierProductPricing;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class ImportFileDataFeed extends DataFeedBaseJob
{
    protected $temporaryTableName;

    protected $dataFeed;

    protected $fieldPath;

    protected $productTable;

    protected $productPricingTable;

    protected $supplierProductPricingTable;

    protected $supplierTable;

    protected $supplierProductsTable;

    protected $useTempTables = false;

    protected $fieldMappings;

    protected $hasSupplierColumn = false;

    protected $pricingTiers = null;

    protected $supplierPricingTiers = null;

    protected $allowedProductColumns = [
        'sku',
        'name',
        'type',
        'barcode',
        'mpn',
        'weight',
        'weight_unit',
        'length',
        'width',
        'height',
        'dimension_unit',
        'fba_prep_instructions',
        'case_quantity',
        'case_length',
        'case_width',
        'case_height',
        'case_dimension_unit',
        'case_weight',
        'case_weight_unit',
        'average_cost',
        'shared_children_attributes',
    ];

    /**
     * Create a new job instance.
     *
     * @return void
     */
    public function __construct($dataFeedId)
    {
        parent::__construct($dataFeedId);
        $this->productTable = (new Product())->getTable();
        $this->supplierTable = (new Supplier())->getTable();
        $this->productPricingTable = (new ProductPricing())->getTable();
        $this->supplierProductPricingTable = (new SupplierProductPricing())->getTable();
        $this->supplierProductsTable = (new SupplierProduct())->getTable();
        $this->pricingTiers = collect([]);
        $this->supplierPricingTiers = collect([]);
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        $this->dataFeed = DataFeed::findOrFail($this->dataFeedId);
        $this->fieldMappings = collect($this->dataFeed->field_settings['field_mappings']);
        $this->hasSupplierColumn = ($this->fieldMappings->where('database_column', 'supplier_name')->count() > 0);

        //Adjust the the pricing Columns
        $this->fieldMappings = $this->fieldMappings->map(function ($column) {
            $columnClone = $column;

            if (Str::contains($column['database_column'], ['price.', '.value'])) {
                $column['is_pricing_column'] = true;
                $column['original_name'] = $column['database_column'];

                //Get Pricing Tier name
                $pricingTierName = str_replace('price.', '', $column['original_name']);
                $pricingTierName = str_replace('.value', '', $pricingTierName);

                //Get Pricing Tier
                $pricingTier = ProductPricingTier::whereName($pricingTierName)->first();

                $column['pricing_tier_name'] = $pricingTierName;
                $column['pricing_tier_id'] = ($pricingTier ? $pricingTier->id : null);

                if ($pricingTier) {
                    $this->pricingTiers->push($pricingTier);
                }
            }

            if (Str::contains($column['database_column'], ['supplier.', '.value'])) {
                $column['is_supplier_pricing_column'] = true;
                $column['original_name'] = $column['database_column'];

                //Get Pricing Tier name
                $supplierPricingTierName = str_replace('supplier.', '', $column['original_name']);
                $supplierPricingTierName = str_replace('.value', '', $supplierPricingTierName);

                //Get Pricing Tier
                $supplierPricingTier = SupplierPricingTier::whereName($supplierPricingTierName)->first();

                $column['supplier_pricing_tier_name'] = $supplierPricingTierName;
                $column['supplier_pricing_tier_id'] = ($supplierPricingTier ? $supplierPricingTier->id : null);

                if ($supplierPricingTier) {
                    $this->supplierPricingTiers->push($supplierPricingTier);
                }
            }

            //Make sure to remove space and dots
            if (@$column['is_pricing_column'] || @$column['is_supplier_pricing_column']) {
                $column['database_column'] = str_replace(' ', '_', $column['database_column']);
                $column['database_column'] = str_replace('.', '_', $column['database_column']);

                $columnClone = $column;
            }

            return $columnClone;
        });

        //Make environment for import
        $this->makeImportEnvironment();

        //Import CSV
        $this->importFile();

        //Upsert Products
        $this->upsertProducts();

        //Make sure supplier exists
        if ($this->hasSupplierColumn) {
            $this->makeSureSupplierExists();
            $this->relateProductsWithSupplier();
        }

        //Update Pricing Tiers
        if ($this->pricingTiers->count() > 0) {
            $this->relateProductsWithPricingTiers();
        }

        //Update Supplier Pricing Tiers
        if ($this->supplierPricingTiers->count() > 0) {
            $this->relateSupplierProductsWithPricingTiers();
        }
    }

    public function relateProductsWithSupplier()
    {
        $query = DB::table($this->temporaryTableName)
            ->selectRaw("distinct $this->supplierTable.id as supplier_id, $this->productTable.id as product_id")
            ->leftJoin($this->supplierTable, "$this->supplierTable.name", '=', "$this->temporaryTableName.supplier_name")
            ->leftJoin($this->productTable, "$this->productTable.sku", '=', "$this->temporaryTableName.sku")
            ->whereRaw("
      $this->temporaryTableName.supplier_name IS NOT NULL
      AND $this->productTable.id IS NOT NULL
      AND $this->supplierTable.id IS NOT NULL
    ");

        //Create Temporary Tables For
        $supplierProductsTemp = $this->supplierProductsTable.'_temp';
        DB::unprepared('CREATE TEMPORARY TABLE '.$supplierProductsTemp.' AS '.$query->toSql());

        //Create insert Query for Suppler Products
        $insertQuery = DB::table($supplierProductsTemp)
            ->selectRaw("distinct $supplierProductsTemp.*, 1 as is_default")
            ->whereNotExists(function ($query) use ($supplierProductsTemp) {
                $query->selectRaw('id')
                    ->from($this->supplierProductsTable)
                    ->whereRaw("$this->supplierProductsTable.product_id = $supplierProductsTemp.product_id")
                    ->whereRaw("$this->supplierProductsTable.supplier_id = $supplierProductsTemp.supplier_id");
            });

        DB::table($this->supplierProductsTable)->insertUsing(['supplier_id', 'product_id', 'is_default'], $insertQuery->toSql());

        //Add Supplier Product ID
        $this->addSupplierProductId();
    }

    public function makeSureSupplierExists()
    {
        //Supplier Check Query
        $supplierQuery = DB::table($this->temporaryTableName)
            ->selectRaw('distinct supplier_name as name')
            ->whereNotExists(function ($query) {
                $query->selectRaw('id')
                    ->from($this->supplierTable)
                    ->whereRaw("$this->supplierTable.name = $this->temporaryTableName.supplier_name");
            });

        //Insert
        DB::table('suppliers')->insertUsing(['name'], $supplierQuery->toSql());

        //Add Supplier ID
        $this->addSupplierId();
    }

    public function makeImportEnvironment()
    {
        $this->setFieldMappins();
        $this->setValidations();
    }

    public function setFieldMappins()
    {
        ImportFile::setFieldMappings($this->fieldMappings);
    }

    public function setValidations()
    {
        //Double Validations
        $double = [
            'weight',
            'length',
            'width',
            'height',
            'case_quantity',
            'case_length',
            'case_height',
            'average_cost',
        ];
        $double = array_merge($this->fieldMappings->where('is_pricing_column', true)->pluck('database_column')->values()->all(), $double);
        ImportFile::setDoubleValidations($double);

        //Set Enum Validations
        ImportFile::setEnumValidations([
            'weight_unit' => "'lb','kg','oz', 'g'",
            'type' => "'standard','bundle','matrix','blemished'",
            'dimension_unit' => "'in','cm', 'mm'",
        ]);

        //Double Validations
        ImportFile::setRequiredValidations([
            'sku',
        ]);
    }

    public function importFile()
    {
        ImportFile::importCsv([
            'file_path' => $this->dataFeed->filePath,
            'is_temp' => $this->useTempTables,
        ]);

        $this->temporaryTableName = ImportFile::getTableName();
    }

    public function insertProducts()
    {
        //Insert query
        $insertQuery = DB::table($this->temporaryTableName)
            ->selectRaw(ImportFile::getDatabaseColumnsImplodeString("$this->temporaryTableName.", $this->allowedProductColumns))
            ->whereNotExists(function ($query) {
                $query->select(DB::raw('id'))
                    ->from($this->productTable)
                    ->whereRaw("$this->productTable.sku = $this->temporaryTableName.sku");
            });

        //Insert Operation
        DB::table($this->productTable)->insertUsing(
            ImportFile::getDatabaseColumns($this->allowedProductColumns)->toArray(),
            $insertQuery->toSql()
        );
    }

    public function updateProducts()
    {
        //Update query
        $updateQuery = DB::table($this->temporaryTableName)
            ->selectRaw(ImportFile::getDatabaseColumnsImplodeString("$this->temporaryTableName.", $this->allowedProductColumns))
            ->whereExists(function ($query) {
                $query->select(DB::raw('id'))
                    ->from($this->productTable)
                    ->whereRaw("$this->productTable.sku = $this->temporaryTableName.sku");
            });

        //Update Operation
        DB::unprepared("UPDATE $this->productTable
        INNER JOIN (".$updateQuery->toSql().") AS stage ON stage.sku = $this->productTable.sku
        SET ".ImportFile::getDatabaseColumnsForUpdateOperation($this->productTable, 'stage.', $this->allowedProductColumns));
    }

    public function addSupplierProductId()
    {
        ImportFile::addColumns([
            [
                'column' => 'supplier_product_id',
                'type' => 'integer',
            ],
        ]);

        DB::unprepared("UPDATE $this->temporaryTableName
                    INNER JOIN $this->supplierProductsTable ON $this->temporaryTableName.product_id = $this->supplierProductsTable.product_id AND $this->temporaryTableName.supplier_id = $this->supplierProductsTable.supplier_id
                    SET supplier_product_id = $this->supplierProductsTable.id");
    }

    public function addSupplierId()
    {
        ImportFile::addColumns([
            [
                'column' => 'supplier_id',
                'type' => 'integer',
            ],
        ]);

        DB::unprepared("UPDATE $this->temporaryTableName
                    LEFT JOIN $this->supplierTable ON $this->temporaryTableName.supplier_name = $this->supplierTable.name
                    SET supplier_id = $this->supplierTable.id");
    }

    public function addProductId()
    {
        ImportFile::addColumns([
            [
                'column' => 'product_id',
                'type' => 'integer',
            ],
        ]);

        DB::unprepared("UPDATE $this->temporaryTableName 
                    LEFT JOIN $this->productTable ON $this->temporaryTableName.sku = $this->productTable.sku
                    SET product_id = $this->productTable.id");
    }

    public function upsertProducts()
    {
        $this->insertProducts();
        $this->updateProducts();
        $this->addProductId();
    }

    public function getProductProcingTierSubquery($pricingTier, $type)
    {
        return DB::table($this->temporaryTableName)
            ->selectRaw($pricingTier['pricing_tier_id'].' as product_pricing_tier_id,product_id,'.$pricingTier['database_column'].' as price')
            ->whereNotNull($pricingTier['database_column'])
            ->$type(function ($query) use ($pricingTier) {
                $query->select(DB::raw('id'))
                    ->from($this->productPricingTable)
                    ->whereRaw("$this->productPricingTable.product_id = $this->temporaryTableName.product_id AND $this->productPricingTable.product_pricing_tier_id = ".$pricingTier['pricing_tier_id']);
            });
    }

    public function getSupplierProductProcingTierSubquery($supplierPricingTier, $type)
    {
        return DB::table($this->temporaryTableName)
            ->selectRaw($supplierPricingTier['supplier_pricing_tier_id'].' as supplier_pricing_tier_id,supplier_product_id,'.$supplierPricingTier['database_column'].' as price')
            ->whereNotNull($supplierPricingTier['database_column'])
            ->$type(function ($query) use ($supplierPricingTier) {
                $query->select(DB::raw('id'))
                    ->from($this->supplierProductPricingTable)
                    ->whereRaw("$this->supplierProductPricingTable.supplier_product_id = $this->temporaryTableName.supplier_product_id AND $this->supplierProductPricingTable.supplier_pricing_tier_id = ".$supplierPricingTier['supplier_pricing_tier_id']);
            });
    }

    public function relateProductsWithPricingTiers()
    {
        $this->fieldMappings->where('is_pricing_column', true)->each(function ($pricingTier) {
            //Insert Operation
            $query = $this->getProductProcingTierSubquery($pricingTier, 'whereNotExists');
            DB::table($this->productPricingTable)->insertUsing(
                ['product_pricing_tier_id', 'product_id', 'price'],
                $query->toSql()
            );

            //Update Operation
            $query = $this->getProductProcingTierSubquery($pricingTier, 'whereExists');

            DB::unprepared("
      UPDATE $this->productPricingTable
      INNER JOIN 
        (".$query->toSql().") AS stage 
          ON stage.product_id = $this->productPricingTable.product_id 
          AND stage.product_pricing_tier_id = $this->productPricingTable.product_pricing_tier_id
      SET $this->productPricingTable.price = stage.price, $this->productPricingTable.product_id = stage.product_id, $this->productPricingTable.product_pricing_tier_id = stage.product_pricing_tier_id");
        });
    }

    public function relateSupplierProductsWithPricingTiers()
    {
        $this->fieldMappings->where('is_supplier_pricing_column', true)->each(function ($supplierPricingTier) {
            //Insert Operation
            $query = $this->getSupplierProductProcingTierSubquery($supplierPricingTier, 'whereNotExists');

            DB::table($this->supplierProductPricingTable)->insertUsing(
                ['supplier_pricing_tier_id', 'supplier_product_id', 'price'],
                $query->toSql()
            );

            //Update Operation
            $query = $this->getSupplierProductProcingTierSubquery($supplierPricingTier, 'whereExists');
            DB::unprepared("
      UPDATE $this->supplierProductPricingTable
      INNER JOIN
        (".$query->toSql().") AS stage
          ON stage.supplier_product_id = $this->supplierProductPricingTable.supplier_product_id
          AND stage.supplier_pricing_tier_id = $this->supplierProductPricingTable.supplier_pricing_tier_id
      SET $this->supplierProductPricingTable.price = stage.price, $this->supplierProductPricingTable.supplier_product_id = stage.supplier_product_id, $this->supplierProductPricingTable.supplier_pricing_tier_id = stage.supplier_pricing_tier_id");
        });
    }
}
