<?php

namespace App\Jobs;

use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\Abstractions\Integrations\SalesChannels\AbstractSalesChannelIntegrationInstance;
use App\Abstractions\Integrations\SalesChannels\AbstractSalesChannelProductManager;
use App\Models\IntegrationInstance;
use App\Repositories\ProductListingRepository;
use App\Repositories\ProductRepository;
use Illuminate\Bus\Batchable;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

class CacheProductListingQuantityJob implements ShouldQueue
{
    use Batchable, Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * @var ProductRepository
     */
    public ProductRepository $productRepository;
    /**
     * @var ProductListingRepository
     */
    protected ProductListingRepository $listings;
    /**
     * @var AbstractSalesChannelProductManager|null
     */
    protected ?AbstractSalesChannelProductManager $productManager = null;

    /**
     * Create a new job instance.
     *
     * @return void
     */
    public function __construct(
        protected IntegrationInstance|IntegrationInstanceInterface $integrationInstance,
        protected array $productIds = []
    )
    {
        $this->onQueue('syncInventory');
        $this->productRepository = app(ProductRepository::class);
        $this->listings = app(ProductListingRepository::class);

        // This condition is needed since some sales channels (e.g. Shopify)
        // don't yet implement the abstraction. Once they do, we can remove the condition.
        if($this->integrationInstance instanceof AbstractSalesChannelIntegrationInstance){
            $this->productManager = $this->integrationInstance->getProductManager();
        }

    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        // Cache product listings quantities
        if(!$this->productManager){
            // We are not using the abstraction,
            // so we go through the repository directly.
            $this->cacheProductListingsViaRepository();
        } else {
            // We are using the abstraction,
            // so we go through the abstraction.
            $this->productManager->cacheProductListingQuantity(
                productIds: $this->productIds
            );
        }
    }


    private function cacheProductListingsViaRepository(): void
    {
        if($this->integrationInstance->hasInventoryLocations()){
            // Handle inventory locations.
            $this->listings->cacheListingQuantityForInventoryLocations(
                integrationInstance: $this->integrationInstance,
                productIds: $this->productIds
            );

            // Aggregate product inventory quantity from locations.
            $this->listings->cacheProductListingQuantityFromInventoryLocations(
                integrationInstance: $this->integrationInstance,
                productIds: $this->productIds
            );
        } else {
            // Handle product listings quantity.
            $this->listings->cacheProductListingQuantity(
                integrationInstance: $this->integrationInstance,
                productIds: $this->productIds
            );
        }
    }


}
