<?php

namespace App\Jobs;

use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\Abstractions\Integrations\SalesChannels\AbstractSalesChannelProduct;
use App\Models\IntegrationInstance;
use App\Models\ProductListing;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Database\Query\JoinClause;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\DB;

class CacheProductListingPriceJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * Create a new job instance.
     *
     * @return void
     */
    public function __construct(protected IntegrationInstance|IntegrationInstanceInterface $integrationInstance)
    {
        //
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        $masterOfPrice = $this->integrationInstance->getMasterOfPrice();

        if ($masterOfPrice == ProductListing::MASTER_SKU) {
            // update the price from the selected pricing tier
            ProductListing::query()
                ->where('sales_channel_id', $this->integrationInstance->salesChannel->id)
                ->whereNull('master_of_price') // inherited
                ->leftJoin('product_pricing as pp', function (JoinClause $join) {
                    $join->on('pp.product_id', '=', 'product_listings.product_id')
                        ->where('pp.product_pricing_tier_id', $this->integrationInstance->getPricingTierId());
                })->update([
                    'product_listings.price' => DB::raw('pp.price'),
                    /*
                                 * The following line is unnecessary because if there is no product listing level override,
                                 * then we can just get the integration instance pricing tier id when needed
                                 */
                    //'product_listings.product_pricing_tier_id' => $this->integrationInstance->getPricingTierId(),
                ]);
        } elseif ($masterOfPrice == ProductListing::MASTER_NEITHER) {
            // price to null
            ProductListing::query()
                ->where('sales_channel_id', $this->integrationInstance->salesChannel->id)
                ->whereNull('master_of_price') // inherited
                ->update(['price' => null]);
        } else {
            // master of price is sales_channel (amazon, shopify...)
            $productModelPath = $this->integrationInstance->integration->getProductsModelPath();
            $salesChannelProductModel = new $productModelPath;
            $priceField = $salesChannelProductModel instanceof AbstractSalesChannelProduct ? $salesChannelProductModel::getPriceField() : 'price';
            // update the price from the sales channel
            ProductListing::query()
                ->where('sales_channel_id', $this->integrationInstance->salesChannel->id)
                ->whereNull('master_of_price') // inherited
                          // TODO: use the product column in the sales channel model after refactoring
                ->join("{$salesChannelProductModel->getTable()} as scp", "scp.{$salesChannelProductModel->getKeyName()}", '=', 'product_listings.document_id')
                ->update(['product_listings.price' => DB::raw("scp.$priceField")]);
        }
    }
}
