<?php

namespace App\Jobs;

use App\Exceptions\UnableToDeleteRecordsException;
use App\Models\Concerns\Archive;
use App\Models\Concerns\HasFilters;
use App\Models\Contracts\Filterable;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

/**
 * Class BulkDeleteRecords.
 */
class BulkDeleteRecords implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * @var array
     */
    protected $request = [];

    /**
     * @var string|Model
     */
    protected $model;

    /**
     * @var string
     */
    protected $idField = 'id';

    /**
     * @var null
     */
    protected $usedTraits = null;

    /**
     * BulkDeleteRecords constructor.
     */
    public function __construct(string $model, array $request, string $idField = 'id')
    {
        $this->request = $request;
        $this->model = $model;
        $this->idField = $idField;
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        set_time_limit(0);

        $exceptionContext = [];

        $this->model::query()
            ->when(isset($this->request['ids']), function ($query) {
                $query->whereIn('id', array_unique($this->request['ids']));
            })
            ->when(! isset($this->request['ids']), function ($query) {
                $query->filter($this->request);
            })
            ->when(isset($this->request['archived']) && $this->archivable(), function ($query) {
                $query->archived($this->request['archived']);
            })
            ->each(function ($current) use (&$exceptionContext) {
                try {
                    $current->delete();
                } catch (\Throwable $exception) {
                    $exceptionContext[class_basename($exception)][] = $current->id;
                    customlog('bulk-delete-records', $exception->getMessage(), $exception->getTraceAsString(), days: 7);
                }
            });

        if (! empty($exceptionContext)) {
            throw new UnableToDeleteRecordsException($exceptionContext);
        }
    }

    protected function filterable(): bool
    {
        return in_array(Filterable::class, class_implements($this->model)) &&
               in_array(HasFilters::class, $this->getUsedTraits());
    }

    protected function archivable(): bool
    {
        return in_array(Archive::class, $this->getUsedTraits());
    }

    protected function getUsedTraits(): array
    {
        if (is_null($this->usedTraits)) {
            $this->usedTraits = class_uses_recursive($this->model);
        }

        return $this->usedTraits ?? [];
    }
}
