<?php

namespace App\Integrations;

use Automattic\WooCommerce\Client;
use Exception;
use Generator;

class WooCommerce extends Channel
{
    /**
     * WooCommerce Order Status.
     */
    const ORDER_STATUS_ANY = 'any';

    const ORDER_STATUS_PENDING = 'pending';

    const ORDER_STATUS_PROCESSING = 'processing';

    const ORDER_STATUS_ON_HOLD = 'on-hold';

    const ORDER_STATUS_COMPLETED = 'completed';

    const ORDER_STATUS_CANCELLED = 'cancelled';

    const ORDER_STATUS_REFUNDED = 'refunded';

    const ORDER_STATUS_FAILED = 'failed';

    const ORDER_STATUS_TRASH = 'trash';

    // Must be valid endpoint.
    const TEST_ENDPOINT = 'orders/count';

    private $wooCommerce_api;

    private $options;

    public function __construct(array $credentials)
    {
        $this->options = $credentials['options'] ?? [];

        if (empty($credentials['url'])
         || empty($credentials['consumer_key'])
         || empty($credentials['consumer_secret'])
        ) {
            throw new Exception('url, consumer_key and consumer_secret are required for WooCommerce sales channel.');
        }
        $this->wooCommerce_api = new Client(
            $credentials['url'],
            $credentials['consumer_key'],
            $credentials['consumer_secret'],
            $this->options
        );
    }

    /**
     * Check sales channel credentials.
     *
     * @return mixed
     */
    public function checkCredentials()
    {
        try {
            // Test request.
            $this->wooCommerce_api->get(self::TEST_ENDPOINT);

            // If request valid get store info.
            return ['status' => true, 'response' => $this->wooCommerce_api->get('')];
        } catch (Exception $exception) {
            switch ($this->wooCommerce_api->http->getResponse()->getCode()) {
                case 401:
                    return [
                        'status' => false,
                        'message' => 'Check your Credentials or Change Authentication type.',
                    ]; // query_string_auth
                case 404:
                    return [
                        'status' => false,
                        'message' => 'Check version and other options.',
                    ]; // version, wp_api
                default:
                    return [
                        'status' => false,
                        'message' => $this->wooCommerce_api->http->getResponse()->getBody() ?: $exception->getMessage(),
                    ];
            }
        }
    }

    /**
     * Get Orders from sales channel.
     *
     * @param  null  $options
     */
    public function getSalesOrders($options = null): Generator
    {
        /**
         * For initials options.
         */
        if (empty($options) || ! is_array($options)) {
            $options = [];
        }

        /**
         * Set request pagination with initial page if not set.
         */
        if (! isset($options['page'])) {
            $initial_page = 1;
            $options['page'] = $initial_page;
        }

        $endpoint = 'orders'; // API Endpoint.
        do {
            try {
                $orders = $this->wooCommerce_api->get($endpoint, $options);

                yield json_decode($this->wooCommerce_api->http->getResponse()->getBody(), true)['orders'];

                // Stop if occur error or orders count zero.
                $hasNext = false;
                if (count($orders->orders)) {
                    $hasNext = true;
                }
            } catch (Exception $exception) {
                yield ['error' => $this->wooCommerce_api->http->getResponse()->getBody() ?: $exception->getMessage()];

                $hasNext = false;
            }

            $options['page'] += 1; // Next page.
        } while ($hasNext);
    }

    /**
     * Get Products from sales channel.
     *
     * @param  null  $options
     */
    public function getProducts($options = null): Generator
    {
        /**
         * For initials options.
         */
        if (empty($options) || ! is_array($options)) {
            $options = [];
        }

        /**
         * Set request pagination with initial page if not set.
         */
        if (! isset($options['page'])) {
            $initial_page = 1;

            $options['page'] = $initial_page;
            $options['per_page'] = 50;
        }

        $endpoint = 'products'; // API Endpoint.
        do {
            try {
                $products = $this->wooCommerce_api->get($endpoint, $options);

                yield json_decode($this->wooCommerce_api->http->getResponse()->getBody(), true)['products'];

                // Stop if occur error or products count zero.
                $hasNext = false;
                if (count($products->products)) {
                    $hasNext = true;
                }
            } catch (Exception $exception) {
                yield ['error' => $this->wooCommerce_api->http->getResponse()->getBody() ?: $exception->getMessage()];

                $hasNext = false;
            }

            $options['page'] += 1; // Next page.
        } while ($hasNext);
    }

    public function getCurrentCurrency()
    {
        $endpoint = '';

        return $this->wooCommerce_api->get($endpoint)->store->meta->currency ?? '';
    }
}
